/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.dbBrowser;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletException;
import javax.portlet.PortletMode;
import javax.portlet.PortletPreferences;
import javax.portlet.PortletRequest;
import javax.portlet.PortletSession;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.ResourceRequest;
import javax.portlet.ResourceResponse;
import javax.sql.DataSource;

import org.apache.commons.dbcp.BasicDataSource;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.portals.bridges.util.PreferencesHelper;
import org.apache.portals.messaging.PortletMessaging;

/**
 * DatabaseBrowserPortlet
 * 
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor </a>
 * @version $Id: DatabaseBrowserPortlet.java 748566 2009-02-27 15:57:03Z vkumar $
 */
public class DatabaseBrowserPortlet extends BrowserPortlet implements Browser
{
    private static final String COLUMNS_TITLE = "columnTitles";
    private static final String EXPORT_TYPE = "exportType";
    private static final String XML_EXPORT = "xmlExport";
    private static final String XLS_EXPORT = "xlsExport";
    protected static final String TABLE_LIST = "tableLists";
    String pathSeprator = System.getProperty("file.separator");
    String pageRoot = System.getProperty("java.io.tmpdir");
    private String[] colNames;
    private SimpleDateFormat date_output;
    private static final String DATE_OUTPUT_PATTERN = "M/d/yyyy HH:mm:ss z";
    BrowserIterator dataBaseIterator;
    private String BLANK_STRING_ARRAY[] = new String[0]; 
    private static final int BLOCK_SIZE = 4096;
    
    public BrowserIterator getLocalIterators(PortletRequest request, String sql, int windowSize) throws Exception
    {
        BrowserIterator iterator;
        List resultSetList = new ArrayList();
        List resultSetTitleList = new ArrayList();
        List resultSetTypeList = new ArrayList();
        Connection con = null;
        PreparedStatement selectStmt = null;
        ResultSet rs = null;
        PortletSession session = request.getPortletSession();
        try
        {
            String poolname = request.getPreferences().getValue(POOLNAME, null);
            if (poolname == null || poolname.length() == 0)
            {
                con = getConnection(request);
            }
            else
            {
                con = getConnection(poolname);
            }
            selectStmt = con.prepareStatement(sql);
            readSqlParameters(request);
            Iterator it = sqlParameters.iterator();
            int ix = 0;
            while (it.hasNext())
            {
                ix++;
                Object object = it.next();
                selectStmt.setObject(ix, object);
            }
            rs = selectStmt.executeQuery();
            ResultSetMetaData rsmd = rs.getMetaData();
            int columnNum = rsmd.getColumnCount();
            /*
             * get the user object types to be displayed and add them to the title list as well as the result set list
             */
            List userObjList = (List) session.getAttribute(USER_OBJECTS);
            int userObjListSize = 0;
            if (userObjList != null)
            {
                userObjListSize = userObjList.size();
            }
            // System.out.println("User List Size = "+ userObjListSize);
            /*SSOProvider
             * the array columnDisplayed maintains a boolean value for each column index. Only the columns that are set to true are added to the resultSetList,
             * resultSetTitleList and resultSetTypeList.
             */
            boolean[] columnDisplayed = new boolean[columnNum + userObjListSize];
            /*
             * this for loop constructs the columnDisplayed array as well as adds to the resultSetTitleList and resultSetTypeList
             */
            for (int i = 1; i <= columnNum; i++)
            {
                int type = rsmd.getColumnType(i);
                if (!((type == Types.BLOB) || (type == Types.CLOB) || (type == Types.BINARY) || (type == Types.LONGVARBINARY) || (type == Types.VARBINARY)))
                {
                    resultSetTitleList.add(rsmd.getColumnName(i));
                    resultSetTypeList.add(String.valueOf(type));
                    columnDisplayed[i - 1] = true;
                }
                else
                {
                    columnDisplayed[i - 1] = false;
                }
            }
            for (int i = columnNum; i < columnNum + userObjListSize; i++)
            {
                ActionParameter usrObj = (ActionParameter) userObjList.get(i - columnNum);
                resultSetTitleList.add(usrObj.getName());
                resultSetTypeList.add(usrObj.getType());
                columnDisplayed[i] = true;
                // System.out.println("User List Name = "+ usrObj.getName()+"
                // Type = "+usrObj.getType());
            }
            /*
             * this while loop adds each row to the resultSetList
             */
            int index = 0;
            while (rs.next())
            {
                List row = new ArrayList(columnNum);
                for (int i = 1; i <= columnNum; i++)
                {
                    if (columnDisplayed[i - 1])
                    {
                        Object obj = rs.getObject(i);
                        if (obj == null)
                        {
                            obj = VELOCITY_NULL_ENTRY;
                        }
                        row.add(obj);
                    }
                }
                for (int i = columnNum; i < columnNum + userObjListSize; i++)
                {
                    ActionParameter usrObj = (ActionParameter) userObjList.get(i - columnNum);
                    if (columnDisplayed[i])
                    {
                        Class c = Class.forName(usrObj.getType());
                        row.add(c.newInstance());
                        populate(index, i, row);
                    }
                }
                resultSetList.add(row);
                index++;
            }
            if (windowSize == -1)
            {
                iterator = new DatabaseBrowserIterator(resultSetList, resultSetTitleList, resultSetTypeList, resultSetList.size());
            }
            else
            {
                iterator = new DatabaseBrowserIterator(resultSetList, resultSetTitleList, resultSetTypeList, windowSize);
            }
        }
        catch (SQLException e)
        {
            throw e;
        }
        finally
        {
            try
            {
                if (null != selectStmt)
                    selectStmt.close();
                if (null != rs)
                    rs.close();
                if (null != con) // closes con also
                {
                    closeConnection(con);
                }
            }
            catch (Exception e)
            {
                throw e;
            }
        }
        return iterator;
    }

    /**
     * Execute the sql statement as specified by the user or the default, and store the resultSet in a vector.
     * 
     * @param sqlt
     *            The sql statement to be executed.
     * @param data
     *            The turbine rundata context for this request.
     */
    public void getRows(RenderRequest request, String sql, int windowSize) throws Exception
    {
        List resultSetList = new ArrayList();
        List resultSetTitleList = new ArrayList();
        List resultSetTypeList = new ArrayList();
        Connection con = null;
        PreparedStatement selectStmt = null;
        ResultSet rs = null;
        PortletSession session = request.getPortletSession();
        try
        {
            String poolname = getPreference(request, POOLNAME, null);
            if (poolname == null || poolname.length() == 0)
            {
                con = getConnection(request);
            }
            else
            {
                con = getConnection(poolname);
            }
            selectStmt = con.prepareStatement(sql);
            readSqlParameters(request);
            Iterator it = sqlParameters.iterator();
            int ix = 0;
            while (it.hasNext())
            {
                ix++;
                Object object = it.next();
                selectStmt.setObject(ix, object);
            }
            rs = selectStmt.executeQuery();
            ResultSetMetaData rsmd = rs.getMetaData();
            int columnNum = rsmd.getColumnCount();
            /*
             * get the user object types to be displayed and add them to the title list as well as the result set list
             */
            List userObjList = (List) session.getAttribute(USER_OBJECTS);
            int userObjListSize = 0;
            if (userObjList != null)
            {
                userObjListSize = userObjList.size();
            }
            // System.out.println("User List Size = "+ userObjListSize);
            /*
             * the array columnDisplayed maintains a boolean value for each column index. Only the columns that are set to true are added to the resultSetList,
             * resultSetTitleList and resultSetTypeList.
             */
            boolean[] columnDisplayed = new boolean[columnNum + userObjListSize];
            /*
             * this for loop constructs the columnDisplayed array as well as adds to the resultSetTitleList and resultSetTypeList
             */
            for (int i = 1; i <= columnNum; i++)
            {
                int type = rsmd.getColumnType(i);
                if (!((type == Types.BLOB) || (type == Types.CLOB) || (type == Types.BINARY) || (type == Types.LONGVARBINARY) || (type == Types.VARBINARY)))
                {
                    resultSetTitleList.add(rsmd.getColumnName(i));
                    resultSetTypeList.add(String.valueOf(type));
                    columnDisplayed[i - 1] = true;
                }
                else
                {
                    columnDisplayed[i - 1] = false;
                }
            }
            for (int i = columnNum; i < columnNum + userObjListSize; i++)
            {
                ActionParameter usrObj = (ActionParameter) userObjList.get(i - columnNum);
                resultSetTitleList.add(usrObj.getName());
                resultSetTypeList.add(usrObj.getType());
                columnDisplayed[i] = true;
                // System.out.println("User List Name = "+ usrObj.getName()+"
                // Type = "+usrObj.getType());
            }
            /*
             * this while loop adds each row to the resultSetList
             */
            int index = 0;
            while (rs.next())
            {
                List row = new ArrayList(columnNum);
                for (int i = 1; i <= columnNum; i++)
                {
                    if (columnDisplayed[i - 1])
                    {
                        Object obj = rs.getObject(i);
                        if (obj == null)
                        {
                            obj = VELOCITY_NULL_ENTRY;
                        }
                        row.add(obj);
                    }
                }
                for (int i = columnNum; i < columnNum + userObjListSize; i++)
                {
                    ActionParameter usrObj = (ActionParameter) userObjList.get(i - columnNum);
                    if (columnDisplayed[i])
                    {
                        Class c = Class.forName(usrObj.getType());
                        row.add(c.newInstance());
                        populate(index, i, row);
                    }
                }
                if (filter(row, request))
                { 
                    continue;
                }
                resultSetList.add(row);
                index++;
            }
            BrowserIterator iterator = new DatabaseBrowserIterator(resultSetList, resultSetTitleList, resultSetTypeList, windowSize);
            setBrowserIterator(request, iterator);
        }
        catch (SQLException e)
        {
            throw e;
        }
        finally
        {
            try
            {
                if (null != selectStmt)
                    selectStmt.close();
                if (null != rs)
                    rs.close();
                if (null != con) // closes con also
                {
                    closeConnection(con);
                }
            }
            catch (Exception e)
            {
                throw e;
            }
        }
    }

    /*
     * Connection Management
     */
    public Connection getConnection(PortletRequest request) throws Exception
    {
        Connection con = null;
        try
        {
            PortletPreferences prefs = request.getPreferences();
            String dsType = prefs.getValue("DatasourceType", null);
            if (dsType == null)
            {
                throw new SQLException("No DataSource provided");
            }
            if (dsType.equals("jndi"))
            {
                Context ctx = new InitialContext();
                String dsName = prefs.getValue("JndiDatasource", "");
                Context envContext = (Context) ctx.lookup("java:/comp/env");
                DataSource ds = (DataSource) envContext.lookup(dsName);
                con = ds.getConnection();
            }
            else if (dsType.equals("dbcp"))
            {
                String url = prefs.getValue("JdbcConnection", "");
                if (url.equals("jdbc:derby:JETSPEED-DEMO"))
                {
                    // prefs.getValue("JdbcConnection"
                    String dbpath = this.getPortletContext().getRealPath("/WEB-INF/demodb");
                    url = "jdbc:derby:" + dbpath;
                }
                BasicDataSource ds = new BasicDataSource();
                ds.setDriverClassName(prefs.getValue("JdbcDriver", ""));
                ds.setUrl(url);
                ds.setUsername(prefs.getValue("JdbcUsername", ""));
                ds.setPassword(prefs.getValue("JdbcPassword", ""));
                // ds.setUrl("jdbc:mysql://j2-server/j2");
                con = ds.getConnection();
            }
            else
            {
                throw new SQLException("No DataSource provided");
            }
        }
        catch (Exception e)
        {
            throw new Exception("Failed to connect", e); // TODO: complete this
        }
        return con;
    }

    private Method[] getResultMethods(List types)
    {
        Method beanMethods[] = new Method[types.size()];
        int type;
        for (int index = 0; index < beanMethods.length; index++)
        {
            type = Integer.parseInt((String) types.get(index));
            switch (type)
            {
                case Types.DATE:
                    //beanMethods[index] = .findMethod(ResultSet.class, "getDate", new Class[] { String.class });
                    break;
                case Types.VARCHAR:
                    //beanMethods[index] = ReflectionUtils.findMethod(ResultSet.class, "getString", new Class[] { String.class });
                    break;
                case Types.CHAR:
                    //beanMethods[index] = ReflectionUtils.findMethod(ResultSet.class, "getString", new Class[] { String.class });
                    break;
                case Types.INTEGER:
                    //beanMethods[index] = ReflectionUtils.findMethod(ResultSet.class, "getInt", new Class[] { String.class });
                    break;
                case Types.FLOAT:
                    //beanMethods[index] = ReflectionUtils.findMethod(ResultSet.class, "getFloat", new Class[] { String.class });
                    break;
                case Types.DOUBLE:
                    //beanMethods[index] = ReflectionUtils.findMethod(ResultSet.class, "getDouble", new Class[] { String.class });
                    break;
            }
        }
        return beanMethods;
    }

    public Connection getConnection(String poolName)
    {
        return null;
    }

    public void closeConnection(Connection con)
    {
        try
        {
            if (con != null)
                con.close();
        }
        catch (SQLException e)
        {
            log.error("Cant close connection", e);
        }
    }

    public void serveResource(ResourceRequest request, ResourceResponse response)
            throws PortletException, IOException
    {
        String exportParam = request.getParameter(EXPORT_TYPE);
        String xlsFileName = "thedata.xls";
        String fileName = "";
        if (!StringUtils.isEmpty(exportParam))
        {
            if (exportParam.equals(XLS_EXPORT))
            {
                try
                {
                    dataBaseIterator = getLocalIterators(request, getQueryString(request), -1);
                    HSSFWorkbook excelWorkbook = generateExcel(dataBaseIterator.getResultSet(), getColumNames(request),
                                                               (String[]) getTitleList(request).toArray(new String[getTitleList(request).size()]),
                                                               getResultMethods(dataBaseIterator.getResultSetTypesList()));
                    if (excelWorkbook != null)
                    {
                        response.setContentType("application/octet-stream");
                        response.setProperty("Content-Disposition", "attachment; filename=export.xls");                        
                        response.getPortletOutputStream().write(excelWorkbook.getBytes());
                    }
                }
                catch (Exception e)
                {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
            }
            else if (exportParam.equals(XML_EXPORT))
            {
            }
       
            /*RequestContext jsReqContext = (RequestContext) request.getAttribute(PortalReservedParameters.REQUEST_CONTEXT_ATTRIBUTE);
            response.sendRedirect(request.getContextPath().replace(
                                                                   request.getContextPath(),
                                                                   jsReqContext.getPortalURL().getBasePath().subSequence(
                                                                                                                         0,
                                                                                                                         jsReqContext.getPortalURL()
                                                                                                                                     .getBasePath()
                                                                                                                                     .lastIndexOf("/"))) +
                                  "/fileserver/content?file=" + userName + "_" + fileName);*/

        }
    }

    public void doEdit(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        Connection con = null;
        response.setContentType("text/html");
        List tableList = null;
        StatusMessage msg = (StatusMessage) PortletMessaging.consume(request, "DatabaseBrowserPortlet", "dbConnectTest");
        if (msg != null)
        {
            this.getContext(request).put("statusMsg", msg);
        }
        String editAction = request.getParameter("configPage");
        if (editAction != null && editAction.equals("configPage1"))
        {
            try
            {
                con = getConnection(request);
                fillColumnDropDown(con, request);
                getContext(request).put("cols", getColNames());
                getContext(request).put("colTitle", getTitleList(request));
                getContext(request).put("colName", getColumNames(request));
                getContext(request).put("sortCols", getSortColums(request));
            }
            catch (Exception e)
            {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
            finally
            {
                this.closeConnection(con);
            }
            request.setAttribute("EditPage", "/WEB-INF/view/database-edit1.vm");
        }
        else
        {
            try
            {
                con = getConnection(request);
                tableList = getTables(request, con);
            }
            catch (Exception e)
            {
                // TODO: handle exception
            }
            if (tableList != null && tableList.size() > 0)
            {
                getContext(request).put(TABLE_LIST, tableList);
            }
        }
        super.doEdit(request, response);
    }

    public void processAction(ActionRequest request, ActionResponse response) throws PortletException, IOException
    {
        String userName = "";
        if (request.getPortletMode() == PortletMode.EDIT)
        {
            String test = request.getParameter("Test");
            if (test != null && test.equals("Test"))
            {
                Connection con = null;
                try
                {
                    PortletPreferences prefs = request.getPreferences();
                    PreferencesHelper.requestParamsToPreferences(request);
                    prefs.store();
                    con = getConnection(request);
                    StatusMessage msg = new StatusMessage("Connection made successfully.", StatusMessage.SUCCESS);
                    PortletMessaging.publish(request, "DatabaseBrowserPortlet", "dbConnectTest", msg);
                }
                catch (Exception e)
                {
                    String msg = e.toString();
                    Throwable cause = e.getCause();
                    if (cause != null)
                    {
                        msg = msg + ", " + cause.getMessage();
                    }
                    StatusMessage sm = new StatusMessage(msg, StatusMessage.ERROR);
                    PortletMessaging.publish(request, "DatabaseBrowserPortlet", "dbConnectTest", sm);
                }
                finally
                {
                    this.closeConnection(con);
                }
                response.setPortletMode(PortletMode.EDIT);
                return;
            }
            String editAction = request.getParameter("configPage");
            if (editAction != null)
            {
                if (editAction.equals("configPage1"))
                {
                    PortletPreferences prefs = request.getPreferences();
                    PreferencesHelper.requestParamsToPreferences(request);
                    prefs.store();
                    response.setPortletMode(PortletMode.EDIT);
                    response.setRenderParameter("configPage", editAction);
                    response.setRenderParameter(TABLE_NAME, request.getParameter(TABLE_NAME));
                    return;
                }
                else if (editAction.equals("configPage"))
                {
                    response.setPortletMode(PortletMode.EDIT);
                    return;
                }
                else if (editAction.equals("save"))
                {
                    PortletPreferences prefs = request.getPreferences();
                    PreferencesHelper.requestParamsToPreferences(request);
                    prefs.store();                    
                    response.setPortletMode(PortletMode.VIEW);
                }                
            }
        }
        super.processAction(request, response);        
    }
    
    private void drain(InputStream r, OutputStream w) throws IOException
    {
        byte[] bytes = new byte[4096];
        try
        {
            int length = r.read(bytes);
            while (length != -1)
            {
                if (length != 0)
                {
                    w.write(bytes, 0, length);
                }
                length = r.read(bytes);
            }
        }
        finally
        {
            bytes = null;
        }
    }
    private String[] getColNames()
    {
        return colNames;
    }

    public String[] getResultSetTitleList(RenderRequest request, String[] columnNames)
    {
        String[] columnsNames = request.getPreferences().getValues(COLUMNS_TITLE, columnNames);
        return columnsNames;
    }

    private void fillColumnDropDown(Connection con, RenderRequest request)
    {
        String tableName;
        try
        {
            if (!StringUtils.isEmpty(request.getParameter(TABLE_NAME)))
            {
                tableName = request.getParameter(TABLE_NAME);
            }
            else
            {
                tableName = request.getPreferences().getValue(TABLE_NAME, "");
            }
            Statement stmt = con.createStatement();
            ResultSet set = stmt.executeQuery("select * from " + tableName);
            ResultSetMetaData meta = set.getMetaData();
            int ColumnCount = meta.getColumnCount();
            List colums = new ArrayList();
            for (int index = 1; index <= ColumnCount; index++)
            {
                colums.add(meta.getColumnName(index));
            }
            this.colNames = new String[colums.size()];
            colums.toArray((String[]) this.colNames);
        }
        catch (Exception e)
        {
            // TODO: handle exception
        }
    }

    public List getTitleList(PortletRequest request)
    {
        String titles = null;
        titles = request.getPreferences().getValue(COLUMNS_TITLE, titles);
        return Arrays.asList(titles.split(","));
    }

    public String[] getColumNames(PortletRequest request)
    {
        String tempColumn = "";
        tempColumn = request.getPreferences().getValue(COLUMNS, tempColumn);
        if(StringUtils.isEmpty(tempColumn))
        {
            return BLANK_STRING_ARRAY;
        }
        return tempColumn.split(",");
    }

    public String[] getSortColums(PortletRequest request)
    {       
        String tempColumn = "";
        tempColumn = request.getPreferences().getValue(Order_BY_COLUMNS, tempColumn);
        if(StringUtils.isEmpty(tempColumn))
        {
            return BLANK_STRING_ARRAY;
        }
        return tempColumn.split(",");
    }
    
    

    public String getQueryString(PortletRequest request)
    {
        StringBuffer query = new StringBuffer();
        String table = request.getPreferences().getValue(TABLE_NAME, "");
        String condtions = request.getPreferences().getValue(CONDITIONS, "");
        String orderBy = request.getPreferences().getValue(Order_BY_COLUMNS, "");
        String columns = request.getPreferences().getValue(COLUMNS, "");
        if (!StringUtils.isEmpty(columns))
        {
            query.append("select ").append(columns).append(" from ").append(table);
        }
        else
        {
            query.append("select * from ").append(table);
        }
        if (!StringUtils.isEmpty(condtions))
        {
            query.append(" ").append(condtions);
        }
        if (!StringUtils.isEmpty(orderBy))
        {
            query.append(" ORDER BY ").append(orderBy);
        }        
        
        return replacePlaceHolder(query.toString(),request);
    }
    private String replacePlaceHolder(String buffer, PortletRequest request){
        String userName="";
        if(request.getUserPrincipal()!=null)
        {
            userName  = request.getUserPrincipal().getName();    
        }else{
            userName = "guest";
        }        
        buffer = buffer.replace("$USER", "'" + userName + "'");
        return buffer;
    }

    private HSSFWorkbook generateExcel(List beanList, String[] colNames, String[] colDescs, Method[] readMethods)
    {
        date_output = new SimpleDateFormat(DATE_OUTPUT_PATTERN);
        HSSFWorkbook workbook = new HSSFWorkbook();
        HSSFSheet sheet = workbook.createSheet("export");
        addColumnHeaders(sheet, colDescs);
        addColumnValues(sheet, beanList, colNames, readMethods);
        return workbook;
    }

    private void addColumnHeaders(HSSFSheet sheet, String[] colDescs)
    {
        HSSFRow rowHeader = sheet.createRow(0);
        for (int i = 0; i < colDescs.length; i++)
        {
            addColumnValue(rowHeader, colDescs[i], i);
        }
    }

    private void addColumnValues(HSSFSheet sheet, List beanList, String[] colNames, Method[] readMethods)
    {
        int rowIndex = 0;
        Iterator beanListIter = beanList.iterator();
        List tempList = null;
        while (beanListIter.hasNext())
        {
            Object bean = beanListIter.next();
            if (bean != null)
            {
                HSSFRow row = sheet.createRow(rowIndex + 1);
                if (bean instanceof ArrayList)
                {
                    tempList = (List) bean;
                    for (int i = 0; i < tempList.size(); i++)
                    {
                        String colVal = columnValueToString(tempList.get(i));
                        addColumnValue(row, colVal, i);
                    }
                }
                else
                {
                    for (int i = 0; i < colNames.length; i++)
                    {
                        String colName = colNames[i];
                        Method colReadMethod = readMethods[i];
                        String colVal = "";
                        if (colReadMethod != null)
                        {
                            try
                            {
                                Object colValObj = colReadMethod.invoke(bean, new Object[] { colName });
                                colVal = columnValueToString(colValObj);
                            }
                            catch (Exception ex)
                            {
                                colVal = "";
                            }
                        }
                        addColumnValue(row, colVal, i);
                    }
                }
                rowIndex++;
            }
        }
    }

    private void addColumnValue(HSSFRow rowHeader, String value, int index)
    {
        HSSFCell cell = rowHeader.createCell((short) index);
        cell.setEncoding(HSSFCell.ENCODING_UTF_16);
        if (value == null)
            value = "";
        cell.setCellValue(value);
    }

    protected String columnValueToString(Object colValObj)
    {
        String colVal = null;
        if (colValObj == null)
            colVal = "";
        else
        {
            if (colValObj instanceof Date)
            {
                colVal = date_output.format((Date) colValObj);
            }
            else
            {
                colVal = colValObj.toString();
            }
        }
        return colVal;
    }

    private String cleanUserFolder(String userName)
    {
        String folderName = "";
        synchronized (this)
        {
            folderName = getUserFolder(userName, false);
            File dir = new File(folderName);
            if (dir.exists())
                dir.delete();
            dir = new File(folderName);
            dir.mkdir();
        }
        return folderName;
    }

    private String getUserFolder(String userName, boolean fullPath)
    {
        if (pathSeprator == null || pathSeprator.equals(""))
            pathSeprator = "/";
        if (fullPath)
        {
            return userName + pathSeprator;
        }
        else
        {
            return pageRoot + pathSeprator + userName;
        }
    }

    private List<String> getTables(PortletRequest request, Connection conn)
    {
        List<String> list = new ArrayList<String>();
        ResultSet rs = null;
        PreparedStatement stmt = null;        
        try
        {
            String databaseName = conn.getMetaData().getDatabaseProductName();
            databaseName = databaseName.toLowerCase();
            String query = getCatalogQuery(databaseName);
            if (query == null)
            {
                rs = conn.getMetaData().getTables(null, null, "%", null); //new String[] {"TABLE"});
                while (rs.next())
                {
                    String schema = rs.getString("TABLE_SCHEM");
                    if (schema != null && schema.equalsIgnoreCase("SYS"))
                        continue;
                    list.add(rs.getString("TABLE_NAME"));
                }        
                return list;
            }
            else
            {
                stmt = conn.prepareStatement(query);
                rs = stmt.executeQuery();
                while (rs.next())
                {
                    list.add(rs.getString(1));
                }
            }
            
        }
        catch (SQLException e)
        {
            e.printStackTrace();
        }
        finally
        {
            try
            {
                if (stmt != null)
                    stmt.close();
                if (rs != null)
                    rs.close();
            }
            catch (Exception e2) 
            {}
        }
        return list;
    }

    private String getCatalogQuery(String databaseName)
    {
        String dataBaseQuery = null;
        if (databaseName.equals("oracle"))
        {
            dataBaseQuery = "SELECT table_name FROM user_tables";
        }
//        else if (databaseName.equals("mysql"))
//        {
//            dataBaseQuery = "show tables";
//        }
        else if (databaseName.equals("sqlserver"))
        {
            dataBaseQuery = "Select name from sys.objects where type='U'";
        }
        //else if (databaseName.equals("apache derby"))
        return dataBaseQuery;
    }
}