/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.gems.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.LinkedList;
import java.util.List;

import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import org.apache.portals.bridges.common.GenericServletPortlet;

/**
 * FilePortlet
 * 
 * Init Parameters:
 *  useLanguage : boolean, should this portlet use a language fallback algorithm, defaults to false
 *  useFilesystem : boolean, should we look in the file system or in a webapp relative path, defaults to false
 *  folder : string, base location of the folder to start searching in, applies to webapp and filesystem mounts
 *  portalParam : string, the portal parameter passed in containing the name of the file (or path) to serve up
 * 
 * Preferences:
 *   file : string, the name of the file to serve up
 *   
 *  Portal provided render attributes:
 *   file : string, the name of the file to serve up. overriding the preference
 *     
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: FilePortlet.java 759390 2009-03-27 23:21:57Z taylor $
 */
public class FilePortlet extends GenericServletPortlet
{
    private String  defaultPortalParam = "org.apache.jetspeed.Path";
    final private String PATH_SEPARATOR = "/";
    
    public void init(PortletConfig config) throws PortletException
    {
        super.init(config);
    }
    
    private boolean getBooleanFromParameter(String param)
    {
        if (param == null)
            return false;
        return param.equalsIgnoreCase("true");            
    }
    
    class InitParameterState
    {
        boolean useLanguage = false;
        boolean useFileSystem = false;
        String folder;
        String portalParam;
        
        InitParameterState(PortletConfig config)
        {
            this.useLanguage = getBooleanFromParameter(config.getInitParameter("useLanguage"));
            this.useFileSystem = getBooleanFromParameter(config.getInitParameter("useFilesystem"));
            this.folder = config.getInitParameter("folder");
            if (this.folder == null)
                this.folder = "";
            String portalParam = config.getInitParameter("portalParam");
            if (portalParam == null)
                this.portalParam = defaultPortalParam;            
        }
    }
    
    public void doView(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        InitParameterState state = new InitParameterState(this.getPortletConfig());
        String fileName = (String)request.getAttribute(state.portalParam);
        if (fileName == null)
            fileName = request.getPreferences().getValue("file", null);
        if (fileName == null)
        {
            response.setContentType("text/html");
            response.getWriter().println("<p>A source document was not provided.</p>"); // TODO: localize
            return;
        }
        String path = this.concatenatePaths(state.folder, fileName);
        setContentType(path, response);
        List<String> paths = fallback(path, request.getLocale().getLanguage(), state);
        renderFile(response, paths, state);
    }

    protected List<String> fallback(String path, String language, InitParameterState state)
    {
        List<String> paths = new LinkedList<String>();
        if (state.useLanguage)
        {
            String fallbackPath = path;
            File temp = new File(path);
            String parentPath = temp.getParent();
            String name = temp.getName();
            path = concatenatePaths(parentPath, language);
            path = concatenatePaths(path, name);
            paths.add(path);
            paths.add(fallbackPath);
        }
        else
        {
            paths.add(path);
        }
        return paths;
    }

    protected void setContentType(String path, RenderResponse response)
    {
        // Note these content types need to be added to the portlet.xml
        if (path.endsWith(".html"))
        {
            response.setContentType("text/html");
        }
        else if (path.endsWith(".pdf"))
        {
            response.setContentType("application/pdf");
        }
        else if (path.endsWith(".zip"))
        {
            response.setContentType("application/zip");
        }
        else if (path.endsWith(".csv"))
        {
            response.setContentType("text/csv");
        }
        else if (path.endsWith(".xml"))
        {
            response.setContentType("text/xml");
        }
        else if (path.endsWith(".xls"))
        {
            File f = new File(path);
            response.setContentType("application/vnd.ms-excel");
            response.addProperty( "Content-Type", "application/vnd.ms-excel" );
            response.addProperty( "Content-Disposition", "inline; filename=" + f.getName());
        }        
        else if (path.endsWith(".psml") || path.endsWith(".link"))
        {
            response.setContentType("text/xml");
        }
        else
        {
            response.setContentType("text/html");
        }
    }

    protected void renderFile(RenderResponse response, List<String> paths, InitParameterState state) throws PortletException, IOException
    {
        boolean drained = false;
        for (String fileName : paths)
        {
            InputStream is = null;
            try
            {
                if (state.useFileSystem)
                {
                    is = new FileInputStream(fileName);
                }
                else
                {
                    is = getPortletContext().getResourceAsStream(fileName);                
                }
                if (is != null)
                {
                    drain(is, response.getPortletOutputStream());
                    response.getPortletOutputStream().flush();
                    is.close();
                    is = null;
                    drained = true;
                    break;
                }
            }
            catch (Exception e)
            {}
            finally
            {
                if (is != null)
                    is.close();
            }
        }
        if (!drained)
        {
            String fileName = (String) paths.get(0);
            byte[] bytes = ("File " + fileName + " not found.").getBytes();
            response.getPortletOutputStream().write(bytes);
            return;
        }
    }

    static final int BLOCK_SIZE = 4096;

    public static void drain(InputStream r, OutputStream w) throws IOException
    {
        byte[] bytes = new byte[BLOCK_SIZE];
        try
        {
            int length = r.read(bytes);
            while (length != -1)
            {
                if (length != 0)
                {
                    w.write(bytes, 0, length);
                }
                length = r.read(bytes);
            }
        }
        finally
        {
            bytes = null;
        }
    }


    protected String concatenatePaths(String base, String path)
    {
        String result = "";
        if (base == null)
        {
            if (path == null)
            {
                return result;
            }
            return path;
        }
        else
        {
            if (path == null)
            {
                return base;
            }
        }
        if (base.endsWith(PATH_SEPARATOR))
        {
            if (path.startsWith(PATH_SEPARATOR))
            {
                result = base.concat(path.substring(1));
                return result;
            }
        }
        else
        {
            if (!path.startsWith(PATH_SEPARATOR))
            {
                result = base.concat(PATH_SEPARATOR).concat(path);
                return result;
            }
        }
        return base.concat(path);
    }

}