/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.rss;

import java.io.IOException;
import java.util.ListResourceBundle;
import java.util.Locale;
import java.util.ResourceBundle;

import javax.portlet.Portlet;
import javax.portlet.PortletConfig;
import javax.portlet.PortletContext;
import javax.portlet.PortletException;
import javax.portlet.PortletPreferences;
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;

import junit.framework.TestCase;

import org.apache.portals.applications.rss.servlets.SpringInitServlet;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.mock.web.MockServletConfig;
import org.springframework.mock.web.MockServletContext;
import org.springframework.mock.web.portlet.MockPortletConfig;
import org.springframework.mock.web.portlet.MockPortletContext;
import org.springframework.mock.web.portlet.MockPortletPreferences;
import org.springframework.mock.web.portlet.MockRenderRequest;
import org.springframework.mock.web.portlet.MockRenderResponse;

/**
 * TestRSSPortlet
 * 
 * @version $Id: TestRSSPortlet.java 927437 2010-03-25 15:04:46Z woonsan $
 */
public class TestRSSPortlet extends TestCase
{
    private static final String TEST_LINE_ESCAPED = "&lt;em&gt;NOTE: DO NOT DELETE THIS LINE! USED IN TEST CASE!!&lt;/em&gt;";
    private static final String TEST_LINE_UNESCAPED = "<em>NOTE: DO NOT DELETE THIS LINE! USED IN TEST CASE!!</em>";
    
    private ServletContext servletContext;
    private ServletConfig servletConfig;
    private SpringInitServlet springInitServlet;
    
    private PortletContext portletContext;
    private PortletConfig portletConfig;
    private RSSPortlet rssPortlet;
    
    @Override
    public void setUp()
    {
        servletContext = new MockServletContext(new DefaultResourceLoader());
        servletConfig = new MockServletConfig(servletContext);
        ((MockServletConfig) servletConfig).addInitParameter("spring-configuration", "/test-spring-init-servlet-beans.xml");
        
        springInitServlet = new SpringInitServlet();
        
        try
        {
            springInitServlet.init(servletConfig);
        }
        catch (ServletException e)
        {
            fail("Failed to initialize spring init servlet: " + e);
        }
        
        portletContext = new MockPortletContext(new DefaultResourceLoader());
        portletConfig = new MockPortletConfig(portletContext);
        ((MockPortletConfig) portletConfig).addInitParameter("stylesheet", "/test-rss.xsl");
        ResourceBundle bundle = new ListResourceBundle()
        {
            protected Object[][] getContents() 
            {
                return new Object[][] {
                    {"javax.portlet.title", "RSS Portlet"},
                    {"javax.portlet.short-title", "RSS"},
                    {"javax.portlet.keywords", "My RSS Portlet"}
                };
            }
        };
        ((MockPortletConfig) portletConfig).setResourceBundle(Locale.ENGLISH, bundle);
        
        rssPortlet = new RSSPortlet();
        
        try
        {
            rssPortlet.init(portletConfig);
        }
        catch (PortletException e)
        {
            fail("Failed to initialize rss portlet: " + e);
        }
    }
    
    @Override
    public void tearDown()
    {
        rssPortlet.destroy();
        springInitServlet.destroy();
    }
    
    public void testEscapedResult() throws PortletException, IOException
    {
        PortletPreferences preferences = new MockPortletPreferences();
        ((MockPortletPreferences) preferences).setValue("url", new ClassPathResource("/test-rss.xml").getURL().toString());
        ((MockPortletPreferences) preferences).setValue("itemdisplayed", "15");
        ((MockPortletPreferences) preferences).setValue("openinpopup", "true");
        ((MockPortletPreferences) preferences).setValue("showdescription", "true");
        ((MockPortletPreferences) preferences).setValue("showtitle", "true");
        ((MockPortletPreferences) preferences).setValue("showtextinput", "true");
        ((MockPortletPreferences) preferences).setValue("minrefresh", "900");
        ((MockPortletPreferences) preferences).setValue("disableescape", "false");
        
        MockRenderRequest request = new MockRenderRequest(portletContext);
        request.setPreferences(preferences);
        request.addLocale(Locale.ENGLISH);
        MockRenderResponse response = new MockRenderResponse();
        
        rssPortlet.render(request, response);
        rssPortlet.cache.clearCache();
        
        System.out.println("The output:");
        String content = response.getContentAsString();
        System.out.println("" + content);
        
        assertTrue("The test line missing: " + TEST_LINE_ESCAPED, content.contains(TEST_LINE_ESCAPED));
    }
    
    public void testUnescapedResult() throws PortletException, IOException
    {
        PortletPreferences preferences = new MockPortletPreferences();
        ((MockPortletPreferences) preferences).setValue("url", new ClassPathResource("/test-rss.xml").getURL().toString());
        ((MockPortletPreferences) preferences).setValue("itemdisplayed", "15");
        ((MockPortletPreferences) preferences).setValue("openinpopup", "true");
        ((MockPortletPreferences) preferences).setValue("showdescription", "true");
        ((MockPortletPreferences) preferences).setValue("showtitle", "true");
        ((MockPortletPreferences) preferences).setValue("showtextinput", "true");
        ((MockPortletPreferences) preferences).setValue("minrefresh", "900");
        ((MockPortletPreferences) preferences).setValue("disableescape", "true");
        
        MockRenderRequest request = new MockRenderRequest(portletContext);
        request.setPreferences(preferences);
        request.addLocale(Locale.ENGLISH);
        MockRenderResponse response = new MockRenderResponse();
        
        rssPortlet.render(request, response);
        rssPortlet.cache.clearCache();
        
        System.out.println("The output:");
        String content = response.getContentAsString();
        System.out.println("" + content);
        
        assertTrue("The test line missing: " + TEST_LINE_UNESCAPED, content.contains(TEST_LINE_UNESCAPED));
    }
}
