/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.rewriter;

import java.io.IOException;
import java.net.URI;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;
import org.apache.http.client.utils.URIUtils;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.ProxyMappingRegistry;
import org.apache.portals.applications.webcontent2.proxy.RequestContext;
import org.apache.portals.applications.webcontent2.proxy.util.CharArraySegment;
import org.apache.portals.applications.webcontent2.proxy.util.RewriterUtils;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewritingContext;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewritingException;
import org.apache.portals.applications.webcontent2.rewriter.impl.AbstractTextLineContentRewriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default text line based content rewriter implementation with basic link rewriting features.
 * <p>
 * This implementation uses a regular expression (see {@link #LINK_PATTERN}) to find
 * all the DOM elements containing any of <code>href</code>, <code>src</code> and <code>action</code>
 * attributes and rewrite those URI attribute values by a resolved {@link ProxyMapping} from the
 * internal {@link ProxyMappingRegistry} in the current reverse proxy processing context.
 * </p>
 */
public class DefaultReverseProxyTextLineContentRewriter extends AbstractTextLineContentRewriter
{

    private static Logger log = LoggerFactory.getLogger(DefaultReverseProxyTextLineContentRewriter.class);

    /**
     * The regular expression pattern to find and replace URI attribute values in DOM elements.
     */
    protected static final Pattern LINK_PATTERN = 
                    Pattern.compile("(\\s|^)(href\\s*=\\s*|src\\s*=\\s*|action\\s*=\\s*|url\\s*\\(\\s*)((\"([^\"]*)\")|('([^']*)'))", Pattern.CASE_INSENSITIVE);

    /**
     * Zero-argument default constructor.
     */
    public DefaultReverseProxyTextLineContentRewriter()
    {
    }

    /**
     * {@inheritDoc}
     * <p>
     * This implementation uses a regular expression (see {@link #LINK_PATTERN}) to find
     * all the DOM elements containing any of <code>href</code>, <code>src</code> and <code>action</code>
     * attributes and rewrite those URI attribute values (see {@link #rewriteURI(String, ContentRewritingContext)})
     * by a resolved {@link ProxyMapping} from the internal {@link ProxyMappingRegistry}
     * in the current reverse proxy processing context.
     * </p>
     */
    @Override
    protected String rewriteLine(String line, ContentRewritingContext context) throws ContentRewritingException, IOException
    {
        CharSequence segment = new CharArraySegment(line);
        Matcher matcher = null;

        StringBuilder sbLine = new StringBuilder();
        boolean anyFound = false;
        String uri = null;

        for (matcher = LINK_PATTERN.matcher(segment); matcher.find(); )
        {
            anyFound = true;

            sbLine.append(segment.subSequence(0, matcher.start()))
                .append(matcher.group(1))
                .append(matcher.group(2));

            uri = matcher.group(5);

            if (uri != null)
            {
                sbLine.append('"').append(rewriteURI(uri, context)).append('"');
            }
            else
            {
                uri = matcher.group(7);
                sbLine.append('\'').append(rewriteURI(uri, context)).append('\'');
            }

            segment = segment.subSequence(matcher.end(), segment.length());
            matcher.reset(segment);
        }

        if (anyFound)
        {
            sbLine.append(segment);
            return sbLine.toString();
        }
        else
        {
            return line;
        }
    }

    /**
     * Rewrites the given <code>uri</code> by resolving a {@link ProxyMapping} from the internal
     * {@link ProxyMappingRegistry} in the current {@link ContentRewritingContext} and {@link ProxyContext}.
     * @param uri
     * @param context
     * @return
     */
    protected String rewriteURI(String uri, ContentRewritingContext context)
    {
        if (StringUtils.isBlank(uri))
        {
            return uri;
        }

        if (!isRewritableURI(uri))
        {
            return uri;
        }

        URI uriObj = null;

        try
        {
            uriObj = URI.create(uri);
        }
        catch (Exception e)
        {
            log.warn("Invalid uri: '{}'.", uri);
            return uri;
        }

        String scheme = uriObj.getScheme();

        if (scheme != null)
        {
            if (!StringUtils.equalsIgnoreCase(scheme, "http") && !StringUtils.equalsIgnoreCase(scheme, "https"))
            {
                // non http(s) urls are not supported.
                return uri;
            }
        }

        ProxyContext proxyContext = (ProxyContext) context.getAttribute(ProxyContext.class.getName());

        if (proxyContext == null)
        {
            log.warn("ProxyContext not found! No rewriting for '{}'.", uri);
            return uri;
        }

        RequestContext requestContext = proxyContext.getRequestContext();
        ProxyMapping currentMapping = (ProxyMapping) context.getAttribute(ProxyMapping.class.getName());
        ProxyMappingRegistry mappingRegistry = (ProxyMappingRegistry) context.getAttribute(ProxyMappingRegistry.class.getName());

        URI remoteURI = URIUtils.resolve(proxyContext.getRemoteURI(), uriObj);

        String localPath = null;

        if (currentMapping.matchesRemote(remoteURI))
        {
            localPath = currentMapping.resolveLocalFromRemote(remoteURI);
        }
        else
        {
            for (ProxyMapping proxyMapping : mappingRegistry.getProxyMappings())
            {
                if (currentMapping == proxyMapping)
                {
                    continue;
                }

                if (proxyMapping.matchesRemote(remoteURI))
                {
                    localPath = proxyMapping.resolveLocalFromRemote(remoteURI);
                    break;
                }
            }
        }

        if (localPath != null)
        {
            return new StringBuilder(80).append(requestContext.getRequestBasePath()).append(localPath).toString();
        }

        return uri;
    }

    /**
     * Returns true if the given <code>uri</code> can be rewritten by this. Otherwise it returns false.
     * @param uri
     * @return
     */
    protected boolean isRewritableURI(String uri)
    {
        return RewriterUtils.isRewritableURI(uri);
    }
}
