/*
    Copyright 2011 John Layt <john@layt.net>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "kdatecomboboxtest.h"

#include <QtCore/QDate>

#include <QtTest/QtTest>
#include <QLineEdit>
#include "kdatecombobox.h"

QTEST_MAIN(KDateComboBoxTest)

void KDateComboBoxTest::testDefaults()
{
    m_combo = new KDateComboBox(nullptr);
    QCOMPARE(m_combo->date(), QDate::currentDate());
    // Missing support in QLocale;
    //QCOMPARE(m_combo->minimumDate(), KLocale::global()->calendar()->earliestValidDate());
    //QCOMPARE(m_combo->maximumDate(), KLocale::global()->calendar()->latestValidDate());
    QCOMPARE(m_combo->isValid(), true);
    QCOMPARE(m_combo->isNull(), false);
    QCOMPARE(m_combo->options(), KDateComboBox::EditDate | KDateComboBox::SelectDate | KDateComboBox::DatePicker | KDateComboBox::DateKeywords);
    QCOMPARE(m_combo->displayFormat(), QLocale::ShortFormat);
    delete m_combo;
}

void KDateComboBoxTest::testValidNull()
{
    m_combo = new KDateComboBox(nullptr);
    QCOMPARE(m_combo->isValid(), true);
    QCOMPARE(m_combo->isNull(), false);
    m_combo->setDate(QDate());
    QCOMPARE(m_combo->isValid(), false);
    QCOMPARE(m_combo->isNull(), true);
    m_combo->setDate(QDate(2000, 1, 1));
    m_combo->lineEdit()->setText(QStringLiteral("invalid"));
    QCOMPARE(m_combo->isValid(), false);
    QCOMPARE(m_combo->isNull(), false);
    delete m_combo;
}

void KDateComboBoxTest::testDateRange()
{
    m_combo = new KDateComboBox(nullptr);
    m_combo->setDate(QDate(2000, 1, 1));
    // Missing support in QLocale;
    //QCOMPARE(m_combo->minimumDate(), KLocale::global()->calendar()->earliestValidDate());
    //QCOMPARE(m_combo->maximumDate(), KLocale::global()->calendar()->latestValidDate());
    QCOMPARE(m_combo->isValid(), true);

    m_combo->setDateRange(QDate(2001, 1, 1), QDate(2002, 1, 1));
    QCOMPARE(m_combo->minimumDate(), QDate(2001, 1, 1));
    QCOMPARE(m_combo->maximumDate(), QDate(2002, 1, 1));
    QCOMPARE(m_combo->isValid(), false);

    m_combo->setDate(QDate(2003, 1, 1));
    QCOMPARE(m_combo->isValid(), false);
    m_combo->setDate(QDate(2001, 6, 1));
    QCOMPARE(m_combo->isValid(), true);
    m_combo->setDate(QDate(2001, 1, 1));
    QCOMPARE(m_combo->isValid(), true);
    m_combo->setDate(QDate(2002, 1, 1));
    QCOMPARE(m_combo->isValid(), true);
    m_combo->setDate(QDate(2000, 12, 31));
    QCOMPARE(m_combo->isValid(), false);
    m_combo->setDate(QDate(2002, 1, 2));
    QCOMPARE(m_combo->isValid(), false);

    m_combo->setDateRange(QDate(1995, 1, 1), QDate(1990, 1, 1));
    QCOMPARE(m_combo->minimumDate(), QDate(2001, 1, 1));
    QCOMPARE(m_combo->maximumDate(), QDate(2002, 1, 1));

    m_combo->setMinimumDate(QDate(2000, 1, 1));
    QCOMPARE(m_combo->minimumDate(), QDate(2000, 1, 1));
    QCOMPARE(m_combo->maximumDate(), QDate(2002, 1, 1));

    m_combo->setMaximumDate(QDate(2003, 1, 1));
    QCOMPARE(m_combo->minimumDate(), QDate(2000, 1, 1));
    QCOMPARE(m_combo->maximumDate(), QDate(2003, 1, 1));

    m_combo->resetDateRange();
    QVERIFY(!m_combo->minimumDate().isValid());
    QVERIFY(!m_combo->maximumDate().isValid());

    // Check functioning when the minimum or maximum date is not already set

    m_combo->setMinimumDate(QDate(2000, 1, 1));
    QCOMPARE(m_combo->minimumDate(), QDate(2000, 1, 1));
    QVERIFY(!m_combo->maximumDate().isValid());

    m_combo->resetMinimumDate();
    QVERIFY(!m_combo->minimumDate().isValid());
    QVERIFY(!m_combo->maximumDate().isValid());

    m_combo->setMaximumDate(QDate(2003, 1, 1));
    QVERIFY(!m_combo->minimumDate().isValid());
    QCOMPARE(m_combo->maximumDate(), QDate(2003, 1, 1));

    m_combo->resetMaximumDate();
    QVERIFY(!m_combo->minimumDate().isValid());
    QVERIFY(!m_combo->maximumDate().isValid());

    delete m_combo;
}

void KDateComboBoxTest::testDateList()
{
    m_combo = new KDateComboBox(nullptr);
    QMap<QDate, QString> map;

    // Test default map
    QCOMPARE(m_combo->dateMap(), map);

    // Test basic map
    map.clear();
    map.insert(QDate(2000, 1, 1), QStringLiteral("New Years Day"));
    map.insert(QDate(2000, 1, 2), QString());
    map.insert(QDate(2000, 1, 3), QStringLiteral("separator"));
    map.insert(QDate(), QStringLiteral("No Date"));
    m_combo->setDateMap(map);
    QCOMPARE(m_combo->dateMap(), map);

    delete m_combo;
}

void KDateComboBoxTest::testOptions()
{
    m_combo = new KDateComboBox(nullptr);
    KDateComboBox::Options options = KDateComboBox::EditDate | KDateComboBox::SelectDate | KDateComboBox::DatePicker | KDateComboBox::DateKeywords;
    QCOMPARE(m_combo->options(), options);
    options = KDateComboBox::EditDate | KDateComboBox::WarnOnInvalid;
    m_combo->setOptions(options);
    QCOMPARE(m_combo->options(), options);
    delete m_combo;
}

void KDateComboBoxTest::testDisplayFormat()
{
    m_combo = new KDateComboBox(nullptr);
    QLocale::FormatType format = QLocale::ShortFormat;
    QCOMPARE(m_combo->displayFormat(), format);
    format = QLocale::NarrowFormat;
    m_combo->setDisplayFormat(format);
    QCOMPARE(m_combo->displayFormat(), format);
    delete m_combo;
}
