// Copyright The OpenTelemetry Authors
// SPDX-License-Identifier: Apache-2.0

package featuregate // import "go.opentelemetry.io/collector/featuregate"

import (
	"flag"
	"strings"

	"go.uber.org/multierr"
)

const (
	featureGatesFlag            = "feature-gates"
	featureGatesFlagDescription = "Comma-delimited list of feature gate identifiers. Prefix with '-' to disable the feature. '+' or no prefix will enable the feature."
)

// NewFlag returns a flag.Value that directly applies feature gate statuses to a Registry.
// Deprecated: Use Registry's RegisterFlags method instead.
func NewFlag(reg *Registry) flag.Value {
	return newFeatureGateValue(reg)
}

func newFeatureGateValue(reg *Registry) flag.Value {
	return &flagValue{reg: reg}
}

// RegisterFlagsOption is an option for RegisterFlags.
type RegisterFlagsOption interface {
	private()
}

// RegisterFlags that directly applies feature gate statuses to a Registry.
func (r *Registry) RegisterFlags(flagSet *flag.FlagSet, _ ...RegisterFlagsOption) {
	flagSet.Var(newFeatureGateValue(r), featureGatesFlag, featureGatesFlagDescription)
}

// flagValue implements the flag.Value interface and directly applies feature gate statuses to a Registry.
type flagValue struct {
	reg *Registry
}

func (f *flagValue) String() string {
	var ids []string
	f.reg.VisitAll(func(g *Gate) {
		id := g.ID()
		if !g.IsEnabled() {
			id = "-" + id
		}
		ids = append(ids, id)
	})
	return strings.Join(ids, ",")
}

func (f *flagValue) Set(s string) error {
	if s == "" {
		return nil
	}

	var errs error
	ids := strings.Split(s, ",")
	for i := range ids {
		id := ids[i]
		val := true
		switch id[0] {
		case '-':
			id = id[1:]
			val = false
		case '+':
			id = id[1:]
		}
		errs = multierr.Append(errs, f.reg.Set(id, val))
	}
	return errs
}
