/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: SeaOfGatesCell.java
 *
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.sun.electric.tool.user.dialogs;

import com.sun.electric.database.EditingPreferences;
import com.sun.electric.database.hierarchy.Cell;
import com.sun.electric.technology.ArcProto;
import com.sun.electric.technology.Technology;
import com.sun.electric.tool.Job;
import com.sun.electric.tool.JobException;
import com.sun.electric.tool.routing.SeaOfGates.SeaOfGatesCellParameters;
import com.sun.electric.tool.user.Highlighter;
import com.sun.electric.tool.user.Resources;
import com.sun.electric.tool.user.User;
import com.sun.electric.tool.user.ui.EditWindow;
import com.sun.electric.tool.user.ui.ToolBar;
import com.sun.electric.tool.user.ui.TopLevel;
import com.sun.electric.tool.user.ui.WindowFrame;
import com.sun.electric.util.TextUtils;
import com.sun.electric.util.math.EDimension;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EventListener;
import java.util.Iterator;
import java.util.List;

import javax.swing.DefaultListModel;
import javax.swing.JFrame;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.ListSelectionModel;
import javax.swing.border.TitledBorder;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

/**
 * Class to handle the "Sea-of-Gates Cell Preferences" dialog.
 */
public class SeaOfGatesCell extends EDialog
{
	private SeaOfGatesCellParameters sogp;
	private JList arcsList, gridPointsList;
	private DefaultListModel arcsModel, gridPointsModel;
	private Technology curTech;
	private static SeaOfGatesCell theDialog = null;
	private boolean fixedValuesChanging = false;

	public static void showSeaOfGatesCellDialog()
	{
		if (theDialog == null)
		{
			JFrame jf = null;
			if (TopLevel.isMDIMode())
				jf = TopLevel.getCurrentJFrame();
			theDialog = new SeaOfGatesCell(jf);
		}
		EditWindow wnd = EditWindow.needCurrent();
		Cell cell = wnd.getCell();
		if (cell != null) theDialog.loadDialog(cell);
		theDialog.setVisible(true);
	}

	/** Creates new form Sea-of-Gates Cell Preferences */
	private SeaOfGatesCell(Frame parent)
	{
		super(parent, false);
		initComponents();

		// initialize the list of arcs and technologies
		arcsModel = new DefaultListModel();
		arcsList = new JList(arcsModel);
		arcsList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		arcsList.getSelectionModel().addListSelectionListener(new javax.swing.event.ListSelectionListener () {
            public void valueChanged(javax.swing.event.ListSelectionEvent evt) { arcClicked(); }
		});
		sogArcList.setViewportView(arcsList);
		for(Iterator<Technology> tIt = Technology.getTechnologies(); tIt.hasNext(); )
		{
			Technology tech = tIt.next();
			sogTechList.addItem(tech.getTechName());
		}

        sogGridAllUp.setBorderPainted(false);
        sogGridAllDown.setBorderPainted(false);
        sogGridIndUp.setBorderPainted(false);
        sogGridIndDown.setBorderPainted(false);
        sogGridSpacingUp.setBorderPainted(false);
        sogGridSpacingDown.setBorderPainted(false);
        sogGridOffsetUp.setBorderPainted(false);
        sogGridOffsetDown.setBorderPainted(false);

        gridPointsModel = new DefaultListModel();
		gridPointsList = new JList(gridPointsModel);
		gridPointsList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		gridPointsList.addMouseListener(new MouseAdapter() {
			public void mouseClicked(MouseEvent evt) { gridPointClicked(); }
		});
		sogGridIndScroll.setViewportView(gridPointsList);

		sogTechList.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { techChanged(); }
		});
		sogDrawGrid.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { drawGrid(); }
		});
		sogNoHorVer.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { horVerChanged(); }
		});
		sogVerOddHorEven.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { horVerChanged(); }
		});
		sogHorOddVerEven.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { horVerChanged(); }
		});
		sogGridNone.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { gridTypeChanged(); }
		});
		sogGridFixed.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { gridTypeChanged(); }
		});
		sogGridArbitrary.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { gridTypeChanged(); }
		});
		sogFavorLayer.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { favorClicked(); }
		});
		sogAvoidLayer.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { preventClicked(); }
		});
		sogNoSteinerTrees.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { steinerTreesChanged(); }
		});
		sogShowGrid.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { updateGridDisplay(true); }
		});
		sogGridIndUp.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { moveGridLines(1, false); }
		});
		sogGridIndDown.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { moveGridLines(-1, false); }
		});
		sogGridAllUp.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { moveGridLines(1, true); }
		});
		sogGridAllDown.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { moveGridLines(-1, true); }
		});
		sogGridSpacingUp.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { shiftGridSpacing(1); }
		});
		sogGridSpacingDown.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { shiftGridSpacing(-1); }
		});
		sogGridOffsetUp.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { shiftGridOffset(1); }
		});
		sogGridOffsetDown.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { shiftGridOffset(-1); }
		});
		sogGridIndDelete.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { deleteGridLine(); }
		});
		sogGridIndNew.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent evt) { addGridLine(); }
		});
		sogFixedSpacing.getDocument().addDocumentListener(new FixedGridDocumentListener());
		sogFixedOffset.getDocument().addDocumentListener(new FixedGridDocumentListener());
	}

	private void loadDialog(Cell cell)
	{
		jLabel1.setText("Sea-of-Gates Properties for Cell " + cell.describe(false));
		sogp = new SeaOfGatesCellParameters(cell);

		// see if steiner trees are already done
		sogNoSteinerTrees.setSelected(sogp.isSteinerDone());

		// setup layer preferences
		if (!sogp.isFavorHorVer()) sogNoHorVer.setSelected(true); else
		{
			if (sogp.isHorizontalEven()) sogVerOddHorEven.setSelected(true); else
				sogHorOddVerEven.setSelected(true);
		}

		curTech = Technology.getCurrent();
		sogTechList.setSelectedItem(curTech.getTechName());

		pack();
	}

	private ArcProto getCurrentlySelectedArc()
	{
		String str = (String)arcsList.getSelectedValue();
		ArcProto ap = curTech.findArcProto(str);
		return ap;
	}

	private void steinerTreesChanged()
	{
		sogp.setSteinerDone(sogNoSteinerTrees.isSelected());
		new UpdateSeaOfGatesCell(sogp);
	}

	private void horVerChanged()
	{
		sogp.setFavorHorVer(!sogNoHorVer.isSelected());
		if (!sogNoHorVer.isSelected())
		{
			sogp.setHorizontalEven(sogVerOddHorEven.isSelected());
		}
		updateGridDisplay(false);
		arcClicked();
		new UpdateSeaOfGatesCell(sogp);
	}

	private void favorClicked()
	{
		ArcProto ap = getCurrentlySelectedArc();
		sogp.setFavored(ap, sogFavorLayer.isSelected());
		new UpdateSeaOfGatesCell(sogp);
	}

	private void preventClicked()
	{
		ArcProto ap = getCurrentlySelectedArc();
		sogp.setPrevented(ap, sogAvoidLayer.isSelected());
		new UpdateSeaOfGatesCell(sogp);
	}

	private List<Double> getCoordinates()
	{
		ArcProto ap = getCurrentlySelectedArc();
		String newGrid = sogp.getGrid(ap);
		List<Double> coords = new ArrayList<Double>();
		if (newGrid != null)
		{
			String[] parts = newGrid.split(",");
			for(int i=0; i<parts.length; i++)
			{
				String part = parts[i].trim();
				if (part.length() == 0) continue;
				double val = TextUtils.atof(part);
				coords.add(new Double(val));
			}
		}
		return coords;
	}

	/**
	 * Method called when the user clicks on an arc name in the list.
	 */
	private void arcClicked()
	{
		ArcProto ap = getCurrentlySelectedArc();
		if (ap == null) return;
		sogAvoidLayer.setSelected(sogp.isPrevented(ap));
		sogFavorLayer.setSelected(sogp.isFavored(ap));
		setGridLabel(ap);

		gridPointsModel.clear();
		List<Double> coords = getCoordinates();
		if (coords.size() == 0)
		{
			sogGridNone.setSelected(true);
			setFixedGridEnabled(false);
			setArbitraryGridEnabled(false);
		} else
		{
			if (coords.size() == 2)
			{
				setFixedGridEnabled(true);
				setArbitraryGridEnabled(false);
				sogGridFixed.setSelected(true);
				double v1 = coords.get(0).doubleValue();
				double v2 = coords.get(1).doubleValue();
				fixedValuesChanging = true;
				sogFixedSpacing.setText(TextUtils.formatDouble(v2-v1));
				sogFixedOffset.setText(TextUtils.formatDouble(v1));
				fixedValuesChanging = false;
			} else
			{
				setFixedGridEnabled(false);
				setArbitraryGridEnabled(true);
				sogGridArbitrary.setSelected(true);
				fixedValuesChanging = true;
				sogFixedSpacing.setText("");
				sogFixedOffset.setText("");
				fixedValuesChanging = false;
			}
			loadArbitraryPoints(coords);
		}
		if (gridPointsModel.getSize() > 0) gridPointsList.setSelectedIndex(0); else
			gridPointsList.setSelectedIndex(-1);

		updateGridDisplay(true);
	}

	private void gridTypeChanged()
	{
		if (sogGridNone.isSelected())
		{
			setFixedGridEnabled(false);
			setArbitraryGridEnabled(false);
			fixedValuesChanging = true;
			sogFixedSpacing.setText("");
			sogFixedOffset.setText("");
			fixedValuesChanging = false;
			gridPointsModel.clear();
			gridPointsList.setSelectedIndex(-1);
			saveGridValues();
		} else if (sogGridFixed.isSelected())
		{
			setFixedGridEnabled(true);
			setArbitraryGridEnabled(false);
			List<Double> coords = getCoordinates();
			double v1 = 0, v2 = 0;
			if (coords.size() > 0) v1 = coords.get(0).doubleValue();
			if (coords.size() > 1) v2 = coords.get(1).doubleValue(); else v2 = v1+10;
			fixedValuesChanging = true;
			sogFixedSpacing.setText(TextUtils.formatDouble(v2-v1));
			fixedValuesChanging = false;
			sogFixedOffset.setText(TextUtils.formatDouble(v1));
		} else
		{
			setFixedGridEnabled(false);
			setArbitraryGridEnabled(true);
			List<Double> coords = getCoordinates();
			gridPointsModel.clear();
			loadArbitraryPoints(coords);
			if (gridPointsModel.getSize() > 0) gridPointsList.setSelectedIndex(0); else
				gridPointsList.setSelectedIndex(-1);
			if (coords.size() != 2)
			{
				fixedValuesChanging = true;
				sogFixedSpacing.setText("");
				sogFixedOffset.setText("");
				fixedValuesChanging = false;
			}
		}
	}

	private void loadArbitraryPoints(List<Double> coords)
	{
		double lastVal = 0;
		for(int i=0; i<coords.size(); i++)
		{
			double val = coords.get(i).doubleValue();
			String elVal = TextUtils.formatDouble(val);
			if (i > 0) elVal += " (+" + TextUtils.formatDouble(val - lastVal) + ")";
			gridPointsModel.addElement(elVal);
			lastVal = val;
		}
	}

	private void setFixedGridEnabled(boolean enabled)
	{
		TitledBorder b = (TitledBorder)gridFixed.getBorder();
		b.setTitleColor(enabled ? Color.black : Color.GRAY);
		sogFixedSpacing.setEnabled(enabled);
		sogFixedOffset.setEnabled(enabled);
		jLabel6.setEnabled(enabled);
		jLabel8.setEnabled(enabled);
		sogGridSpacingUp.setEnabled(enabled);
		sogGridSpacingDown.setEnabled(enabled);
		sogGridOffsetUp.setEnabled(enabled);
		sogGridOffsetDown.setEnabled(enabled);
	}

	private void setArbitraryGridEnabled(boolean enabled)
	{
		TitledBorder b = (TitledBorder)gridArbitrary.getBorder();
		b.setTitleColor(enabled ? Color.black : Color.GRAY);
		gridPointsList.setEnabled(enabled);
		jLabel3.setEnabled(enabled);
		jLabel4.setEnabled(enabled);
		sogDrawGrid.setEnabled(enabled);
		sogGridAllUp.setEnabled(enabled);
		sogGridAllDown.setEnabled(enabled);
		sogGridIndUp.setEnabled(enabled);
		sogGridIndDown.setEnabled(enabled);
		sogGridIndDelete.setEnabled(enabled);
		sogGridIndNew.setEnabled(enabled);
	}

	private void fixedGridValuesChanged()
	{
		if (fixedValuesChanging) return;
		double v1 = TextUtils.atof(sogFixedSpacing.getText());
		double v2 = TextUtils.atof(sogFixedOffset.getText());
		gridPointsModel.clear();
		gridPointsModel.addElement(TextUtils.formatDouble(v2));
		gridPointsModel.addElement(TextUtils.formatDouble(v2+v1) + " (+" + TextUtils.formatDouble(v1) + ")");
		saveGridValues();
	}

	/**
	 * Class to handle special changes to changes to fixed-grid values.
	 */
	private class FixedGridDocumentListener implements DocumentListener
	{
		public void changedUpdate(DocumentEvent e) { fixedGridValuesChanged(); }
		public void insertUpdate(DocumentEvent e) { fixedGridValuesChanged(); }
		public void removeUpdate(DocumentEvent e) { fixedGridValuesChanged(); }
	}

	private void shiftGridSpacing(int dir)
	{
		double v1 = TextUtils.atof(sogFixedSpacing.getText()) + dir;
		double v2 = TextUtils.atof(sogFixedOffset.getText());
		fixedValuesChanging = true;
		sogFixedSpacing.setText(TextUtils.formatDouble(v1));
		fixedValuesChanging = false;
		gridPointsModel.clear();
		gridPointsModel.addElement(TextUtils.formatDouble(v2));
		gridPointsModel.addElement(TextUtils.formatDouble(v2+v1) + " (+" + TextUtils.formatDouble(v1) + ")");
		saveGridValues();
	}

	private void shiftGridOffset(int dir)
	{
		double v1 = TextUtils.atof(sogFixedSpacing.getText());
		double v2 = TextUtils.atof(sogFixedOffset.getText()) + dir;
		fixedValuesChanging = true;
		sogFixedOffset.setText(TextUtils.formatDouble(v2));
		fixedValuesChanging = false;
		gridPointsModel.clear();
		gridPointsModel.addElement(TextUtils.formatDouble(v2));
		gridPointsModel.addElement(TextUtils.formatDouble(v2+v1) + " (+" + TextUtils.formatDouble(v1) + ")");
		saveGridValues();
	}

	/**
	 * Method called when the arc changes or when the horizontal/vertical properties change.
	 * Updates the label on the "grid spacing" field.
	 * @param ap the current ArcProto.
	 */
	private void setGridLabel(ArcProto ap)
	{
		Boolean isHor = getGridHorizontal(ap);
		if (isHor == null) sogGridLabel.setText("Grid:"); else
		{
			if (isHor.booleanValue()) sogGridLabel.setText("Hor. Grid:"); else
				sogGridLabel.setText("Vert. Grid:");
		}
	}

	private Boolean getGridHorizontal(ArcProto ap)
	{
		if (sogNoHorVer.isSelected()) return null;
		Boolean isHor = Boolean.TRUE;
		if (sogVerOddHorEven.isSelected())
		{
			if ((ap.getFunction().getLevel()%2) != 0) isHor = Boolean.FALSE;
		} else
		{
			if ((ap.getFunction().getLevel()%2) == 0) isHor = Boolean.FALSE;
		}
		return isHor;
	}

	/**
	 * Method called when the technology (for default arcs) has changed.
	 */
	private void techChanged()
	{
		String techName = (String)sogTechList.getSelectedItem();
		Technology tech = Technology.findTechnology(techName);
		if (tech == null) return;
		curTech = tech;

		arcsModel.clear();
		for(Iterator<ArcProto> it = tech.getArcs(); it.hasNext(); )
		{
			ArcProto ap = it.next();
			if (!ap.getFunction().isMetal()) continue;
			arcsModel.addElement(ap.getName());
		}
		arcsList.setSelectedIndex(0);
		arcClicked();
	}

	private void gridPointClicked()
	{
		updateGridDisplay(true);
	}

	private void moveGridLines(int dir, boolean moveAll)
	{
		EDimension dim = User.getAlignmentToGrid();
		if (moveAll)
		{
			double lastVal = 0;
			for(int i=0; i<gridPointsModel.getSize(); i++)
			{
				double val = TextUtils.atof((String)gridPointsModel.get(i)) + dim.getHeight() * dir;
				String elVal = TextUtils.formatDouble(val);
				if (i > 0) elVal += " (+" + TextUtils.formatDouble(val - lastVal) + ")";
				gridPointsModel.set(i, elVal);
				lastVal = val;
			}
		} else
		{
			int index = gridPointsList.getSelectedIndex();
			double v = TextUtils.atof((String)gridPointsModel.get(index)) + dim.getHeight() * dir;
			gridPointsModel.set(index, v + "");
			sortGridPoints();
		}
		saveGridValues();
		updateGridDisplay(false);
	}

	private void sortGridPoints()
	{
		List<Double> vals = new ArrayList<Double>();
		for(int i=0; i<gridPointsModel.getSize(); i++)
		{
			double v = TextUtils.atof((String)gridPointsModel.get(i));
			vals.add(new Double(v));
		}
		Collections.sort(vals);
		double lastVal = 0;
		for(int i=0; i<gridPointsModel.getSize(); i++)
		{
			double val = vals.get(i).doubleValue();
			String elVal = TextUtils.formatDouble(val);
			if (i > 0) elVal += " (+" + TextUtils.formatDouble(val - lastVal) + ")";
			gridPointsModel.set(i, elVal);
			lastVal = val;
		}		
	}

	private void addGridLine()
	{
		String str = Job.getUserInterface().askForInput("Grid Line Coordinate(s):", "Make New Grid Lines (comma-separated list)", "0");
		if (str == null) return;
		String[] parts = str.split(",");
		for(int i=0; i<parts.length; i++)
		{
			String part = parts[i].trim();
			if (part.length() == 0) continue;
			double vInsert = TextUtils.atof(part);
			insertGridLine(vInsert);
		}
		saveGridValues();
		updateGridDisplay(false);
	}

	private void insertGridLine(double vInsert)
	{
		boolean inserted = false;
		double lastVal = 0;
		for(int i=0; i<gridPointsModel.getSize(); i++)
		{
			double v = TextUtils.atof((String)gridPointsModel.get(i));
			if (vInsert <= v)
			{
				String elVal = TextUtils.formatDouble(vInsert);
				if (i > 0) elVal += " (+" + TextUtils.formatDouble(vInsert - lastVal) + ")";
				gridPointsModel.add(i, elVal);
				if (i < gridPointsModel.getSize()-1)
				{
					double nextV = TextUtils.atof((String)gridPointsModel.get(i+1));
					elVal = TextUtils.formatDouble(nextV) + " (+" + TextUtils.formatDouble(nextV - vInsert) + ")";
					gridPointsModel.add(i+1, elVal);
				}
				gridPointsList.setSelectedIndex(i);
				inserted = true;
				break;
			}
			lastVal = v;
		}
		if (!inserted)
		{
			String elVal = TextUtils.formatDouble(vInsert);
			if (gridPointsModel.getSize() > 0) elVal += " (+" + TextUtils.formatDouble(vInsert - lastVal) + ")";
			gridPointsModel.addElement(elVal);
			gridPointsList.setSelectedIndex(gridPointsModel.getSize()-1);
		}
	}

	private void deleteGridLine()
	{
		int index = gridPointsList.getSelectedIndex();
		if (index < 0) return;
		gridPointsModel.remove(index);
		sortGridPoints();
		if (index >= gridPointsModel.getSize()) index--;
		gridPointsList.setSelectedIndex(index);
		saveGridValues();
		updateGridDisplay(false);
	}

	private void drawGrid()
	{
		EventListener oldListener = WindowFrame.getListener();
		Cursor oldCursor = TopLevel.getCurrentCursor();
		EventListener newListener = oldListener;
		if (newListener == null || !(newListener instanceof PlaceGridLineListener))
		{
			TopLevel.setCurrentCursor(ToolBar.outlineCursor);
			newListener = new PlaceGridLineListener(oldListener, oldCursor, this);
			WindowFrame.setListener(newListener);
		}
	}

	/**
	 * Class to choose a location for new node placement.
	 */
	public static class PlaceGridLineListener
		implements MouseMotionListener, MouseListener, MouseWheelListener, KeyListener
	{
		private EventListener oldListener;
		private Cursor oldCursor;
		private SeaOfGatesCell dialog;

		/**
		 * Places a new grid line.
		 * @param oldListener
		 * @param oldCursor
		 */
		private PlaceGridLineListener(EventListener oldListener, Cursor oldCursor, SeaOfGatesCell dialog)
		{
			this.oldListener = oldListener;
			this.oldCursor = oldCursor;
			this.dialog = dialog;
		}

		public void mouseReleased(MouseEvent evt)
		{
			if (!(evt.getSource() instanceof EditWindow)) return;
			EditWindow wnd = (EditWindow)evt.getSource();

			Cell cell = wnd.getCell();
			if (cell == null)
			{
				JOptionPane.showMessageDialog(TopLevel.getCurrentJFrame(),
					"Cannot create node: this window has no cell in it");
				return;
			}

			// restore the former listener to the edit windows
			finished(wnd);
			ArcProto ap = dialog.getCurrentlySelectedArc();
			Boolean isHor = dialog.getGridHorizontal(ap);
			if (isHor != null)
			{
				Point2D pt = wnd.screenToDatabase(evt.getX(), evt.getY());
				EditWindow.gridAlign(pt);
				double val = isHor.booleanValue() ? pt.getY() : pt.getX();

				// shift to be in the range of the grid locations
				if (dialog.gridPointsModel.getSize() >= 2)
				{
					double lowest = TextUtils.atof((String)dialog.gridPointsModel.get(0));
					double highest = TextUtils.atof((String)dialog.gridPointsModel.get(dialog.gridPointsModel.getSize()-1));
					double range = highest - lowest;
					if (range > 0)
					{
						Rectangle2D bounds = wnd.getDisplayedBounds();
						if (isHor.booleanValue())
						{
							double low = Math.floor((bounds.getMinY() - lowest) / range) * range;
							double high = Math.ceil((bounds.getMaxY() - lowest) / range) * range;
							for(double v = low; v <= high; v += range)
							{
								if (bounds.getCenterY() < v+lowest || bounds.getCenterY() >= v+lowest+range) continue;
								val -= v;
								break;
							}
						} else
						{
							double low = Math.floor((bounds.getMinX() - lowest) / range) * range;
							double high = Math.ceil((bounds.getMaxX() - lowest) / range) * range;
							for(double v = low; v <= high; v += range)
							{
								if (bounds.getCenterX() < v+lowest || bounds.getCenterX() >= v+lowest+range) continue;
								val -= v;
								break;
							}
						}
					}
				}
				dialog.insertGridLine(val);
			}
		}

		private void finished(EditWindow wnd)
		{
			if (wnd != null)
			{
				Highlighter highlighter = wnd.getHighlighter();
				highlighter.clear();
				highlighter.finished();
			}
			WindowFrame.setListener(oldListener);
			TopLevel.setCurrentCursor(oldCursor);
		}

		public void mousePressed(MouseEvent evt) {}
		public void mouseClicked(MouseEvent evt) {}
		public void mouseEntered(MouseEvent evt) {}
		public void mouseExited(MouseEvent evt) {}
		public void mouseWheelMoved(MouseWheelEvent evt) {}

		public void mouseMoved(MouseEvent evt)
		{
			showGridLine(evt);
		}

		public void mouseDragged(MouseEvent evt)
		{
			showGridLine(evt);
		}

		private void showGridLine(MouseEvent evt)
		{
			if (!(evt.getSource() instanceof EditWindow)) return;
			EditWindow wnd = (EditWindow)evt.getSource();
			Cell cell = wnd.getCell();
			Highlighter highlighter = wnd.getHighlighter();
			highlighter.clear();
			Rectangle2D bounds = wnd.getDisplayedBounds();
			Point2D pt = wnd.screenToDatabase(evt.getX(), evt.getY());
			EditWindow.gridAlign(pt);
			boolean drawHor = true, drawVer = true;
			ArcProto ap = dialog.getCurrentlySelectedArc();
			Boolean isHor = dialog.getGridHorizontal(ap);
			if (isHor != null)
			{
				if (isHor.booleanValue()) drawVer = false; else
					drawHor = false;
			}
			if (drawHor)
				highlighter.addLine(new Point2D.Double(bounds.getMinX(), pt.getY()), new Point2D.Double(bounds.getMaxX(), pt.getY()),
					cell, true, Color.RED, false);
			if (drawVer)
				highlighter.addLine(new Point2D.Double(pt.getX(), bounds.getMinY()), new Point2D.Double(pt.getX(), bounds.getMaxY()),
					cell, true, Color.RED, false);
			highlighter.finished();
		}

		public void keyPressed(KeyEvent evt)
		{
			int chr = evt.getKeyCode();
			if (chr == KeyEvent.VK_A || chr == KeyEvent.VK_ESCAPE)
			{
				// abort
				finished(EditWindow.getCurrent());
			}
		}

		public void keyReleased(KeyEvent evt) {}
		public void keyTyped(KeyEvent evt) {}
	}


	/**
	 * Method to save the grid values when they have changed.
	 */
	private void saveGridValues()
	{
		ArcProto ap = getCurrentlySelectedArc();
		String newGrid = "";
		for(int i=0; i<gridPointsModel.getSize(); i++)
		{
			String elVal = (String)gridPointsModel.get(i);
			int spacePos = elVal.indexOf(' ');
			if (spacePos >= 0) elVal = elVal.substring(0, spacePos);
			if (newGrid.length() > 0) newGrid += ",";
			newGrid += elVal;
		}
		sogp.setGrid(ap, newGrid);
		new UpdateSeaOfGatesCell(sogp);
	}

	/**
	 * Method to redraw the grid lines on the screen (if grid display is selected).
	 */
	private void updateGridDisplay(boolean force)
	{
		EditWindow wnd = EditWindow.getCurrent();
		if (wnd == null) return;
		Highlighter h = wnd.getRulerHighlighter();
		h.clear();
		if (sogShowGrid.isSelected())
		{
			boolean hor = true, ver = true;
			ArcProto ap = getCurrentlySelectedArc();
			Boolean isHor = getGridHorizontal(ap);
			if (isHor != null)
			{
				if (isHor.booleanValue()) ver = false; else hor = false;
			}

			if (gridPointsModel.getSize() >= 2)
			{
				Cell cell = wnd.getCell();
				int prev = (int)(TextUtils.atof((String)gridPointsModel.get(0)) * wnd.getScale());
				boolean spreadOut = false;
				for(int i=1; i<gridPointsModel.getSize(); i++)
				{
					int val = (int)(TextUtils.atof((String)gridPointsModel.get(i)) * wnd.getScale());
					if (Math.abs(val-prev) > 3) { spreadOut = true;   break; }
					prev = val;
				}
				if (spreadOut)
				{
					Rectangle2D bounds = wnd.getDisplayedBounds();
					double offset = TextUtils.atof((String)gridPointsModel.get(0));
					double range = TextUtils.atof((String)gridPointsModel.get(gridPointsModel.getSize()-1)) - offset;
					int index = gridPointsList.getSelectedIndex();
					if (range > 0)
					{
						if (hor)
						{
							double low = Math.floor((bounds.getMinY() - offset) / range) * range;
							double high = Math.ceil((bounds.getMaxY() - offset) / range) * range;
							for(double v = low; v <= high; v += range)
							{
								int startInd = 0;
								Color col = Color.WHITE;
								if (bounds.getCenterY() >= v+offset && bounds.getCenterY() < v+offset+range) col = Color.RED; else
									if (v != low) startInd = 1;
								for(int i=startInd; i<gridPointsModel.getSize(); i++)
								{
									double val = TextUtils.atof((String)gridPointsModel.get(i));
									Point2D start = new Point2D.Double(bounds.getMinX(), v + val);
									Point2D end = new Point2D.Double(bounds.getMaxX(), v + val);
									boolean thick = false;
									if (col == Color.red && i == index) thick = true;
									h.addLine(start, end, cell, thick, col, false);
								}
							}
						}
						if (ver)
						{
							double low = Math.floor((bounds.getMinX() - offset) / range) * range;
							double high = Math.ceil((bounds.getMaxX() - offset) / range) * range;
							for(double v = low; v <= high; v += range)
							{
								int startInd = 0;
								Color col = Color.WHITE;
								if (bounds.getCenterX() >= v+offset && bounds.getCenterX() < v+offset+range) col = Color.RED; else
									if (v != low) startInd = 1;
								for(int i=startInd; i<gridPointsModel.getSize(); i++)
								{
									double val = TextUtils.atof((String)gridPointsModel.get(i));
									Point2D start = new Point2D.Double(v + val, bounds.getMinY());
									Point2D end = new Point2D.Double(v + val, bounds.getMaxY());
									boolean thick = false;
									if (col == Color.red && i == index) thick = true;
									h.addLine(start, end, cell, thick, col, false);
								}
							}
						}
					}
				}
			}
		}
		h.finished();
		if (force) wnd.repaint();
	}

	private void done() { setVisible(false); }

	protected void escapePressed() { done(); }

	/**
	 * Class to delete an attribute in a new thread.
	 */
	private static class UpdateSeaOfGatesCell extends Job
	{
		private SeaOfGatesCellParameters sogp;

		private UpdateSeaOfGatesCell(SeaOfGatesCellParameters sogp)
		{
			super("Update Sea-of-Gates Cell Properties", User.getUserTool(), Job.Type.CHANGE, null, null, Job.Priority.USER);
			this.sogp = sogp;
			startJob();
		}

		@Override
		public boolean doIt() throws JobException
		{
			EditingPreferences ep = getEditingPreferences();
			sogp.saveParameters(ep);
			return true;
		}
	}

	/** This method is called from within the constructor to
	 * initialize the form.
	 * WARNING: Do NOT modify this code. The content of this method is
	 * always regenerated by the Form Editor.
	 */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        sogHorVerChoices = new javax.swing.ButtonGroup();
        sogGridType = new javax.swing.ButtonGroup();
        ok = new javax.swing.JButton();
        jLabel1 = new javax.swing.JLabel();
        individualLayers = new javax.swing.JPanel();
        sogArcList = new javax.swing.JScrollPane();
        jLabel2 = new javax.swing.JLabel();
        sogTechList = new javax.swing.JComboBox();
        sogAvoidLayer = new javax.swing.JCheckBox();
        sogFavorLayer = new javax.swing.JCheckBox();
        gridControl = new javax.swing.JPanel();
        sogGridNone = new javax.swing.JRadioButton();
        sogGridFixed = new javax.swing.JRadioButton();
        sogGridArbitrary = new javax.swing.JRadioButton();
        sogShowGrid = new javax.swing.JCheckBox();
        sogGridLabel = new javax.swing.JLabel();
        gridFixed = new javax.swing.JPanel();
        jLabel6 = new javax.swing.JLabel();
        sogFixedSpacing = new javax.swing.JTextField();
        jLabel8 = new javax.swing.JLabel();
        sogFixedOffset = new javax.swing.JTextField();
        sogGridSpacingUp = new javax.swing.JButton();
        sogGridSpacingDown = new javax.swing.JButton();
        sogGridOffsetUp = new javax.swing.JButton();
        sogGridOffsetDown = new javax.swing.JButton();
        gridArbitrary = new javax.swing.JPanel();
        sogDrawGrid = new javax.swing.JButton();
        sogGridIndScroll = new javax.swing.JScrollPane();
        sogGridAllUp = new javax.swing.JButton();
        sogGridAllDown = new javax.swing.JButton();
        sogGridIndUp = new javax.swing.JButton();
        sogGridIndDown = new javax.swing.JButton();
        sogGridIndDelete = new javax.swing.JButton();
        sogGridIndNew = new javax.swing.JButton();
        jLabel3 = new javax.swing.JLabel();
        jLabel4 = new javax.swing.JLabel();
        entireCell = new javax.swing.JPanel();
        sogNoSteinerTrees = new javax.swing.JCheckBox();
        sogHorOddVerEven = new javax.swing.JRadioButton();
        sogVerOddHorEven = new javax.swing.JRadioButton();
        sogNoHorVer = new javax.swing.JRadioButton();
        jLabel7 = new javax.swing.JLabel();

        setTitle("Sea-of-Gates Cell Properties");
        setName(""); // NOI18N
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                closeDialog(evt);
            }
        });
        getContentPane().setLayout(new java.awt.GridBagLayout());

        ok.setText("Done");
        ok.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                ok(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.weightx = 0.5;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        getContentPane().add(ok, gridBagConstraints);

        jLabel1.setText("Sea-of-Gates Properties for Cell XXX");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        getContentPane().add(jLabel1, gridBagConstraints);

        individualLayers.setBorder(javax.swing.BorderFactory.createTitledBorder("For Individual Layers in the Cell"));
        individualLayers.setLayout(new java.awt.GridBagLayout());

        sogArcList.setMinimumSize(new java.awt.Dimension(24, 100));
        sogArcList.setPreferredSize(new java.awt.Dimension(100, 100));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 0.5;
        individualLayers.add(sogArcList, gridBagConstraints);

        jLabel2.setText("Technology:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.weightx = 0.5;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        individualLayers.add(jLabel2, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.5;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        individualLayers.add(sogTechList, gridBagConstraints);

        sogAvoidLayer.setText("Avoid layer use");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.weightx = 0.5;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        individualLayers.add(sogAvoidLayer, gridBagConstraints);

        sogFavorLayer.setText("Favor layer");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.weightx = 0.5;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        individualLayers.add(sogFavorLayer, gridBagConstraints);

        gridControl.setLayout(new java.awt.GridBagLayout());

        sogGridType.add(sogGridNone);
        sogGridNone.setText("None");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridControl.add(sogGridNone, gridBagConstraints);

        sogGridType.add(sogGridFixed);
        sogGridFixed.setText("Fixed");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridControl.add(sogGridFixed, gridBagConstraints);

        sogGridType.add(sogGridArbitrary);
        sogGridArbitrary.setText("Arbitrary");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridControl.add(sogGridArbitrary, gridBagConstraints);

        sogShowGrid.setText("Show");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 0, 4);
        gridControl.add(sogShowGrid, gridBagConstraints);

        sogGridLabel.setText("Grid:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridControl.add(sogGridLabel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.weightx = 1.0;
        individualLayers.add(gridControl, gridBagConstraints);

        gridFixed.setBorder(javax.swing.BorderFactory.createTitledBorder("Fixed Grid Control"));
        gridFixed.setLayout(new java.awt.GridBagLayout());

        jLabel6.setText("Spacing:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridheight = 2;
        gridFixed.add(jLabel6, gridBagConstraints);

        sogFixedSpacing.setColumns(7);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridheight = 2;
        gridFixed.add(sogFixedSpacing, gridBagConstraints);

        jLabel8.setText("Offset:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridheight = 2;
        gridBagConstraints.insets = new java.awt.Insets(0, 20, 0, 0);
        gridFixed.add(jLabel8, gridBagConstraints);

        sogFixedOffset.setColumns(7);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridheight = 2;
        gridFixed.add(sogFixedOffset, gridBagConstraints);

        sogGridSpacingUp.setIcon(Resources.getResource(getClass(), "IconIncrement.gif"));
        sogGridSpacingUp.setToolTipText("Move all grid lines up or right");
        sogGridSpacingUp.setMargin(new java.awt.Insets(0, 0, 0, 0));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 0, 4);
        gridFixed.add(sogGridSpacingUp, gridBagConstraints);

        sogGridSpacingDown.setIcon(Resources.getResource(getClass(), "IconDecrement.gif"));
        sogGridSpacingDown.setToolTipText("Move all grid lines down or left");
        sogGridSpacingDown.setMargin(new java.awt.Insets(0, 0, 0, 0));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 4, 4);
        gridFixed.add(sogGridSpacingDown, gridBagConstraints);

        sogGridOffsetUp.setIcon(Resources.getResource(getClass(), "IconIncrement.gif"));
        sogGridOffsetUp.setToolTipText("Move grid line up or right");
        sogGridOffsetUp.setMargin(new java.awt.Insets(0, 0, 0, 0));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 0, 4);
        gridFixed.add(sogGridOffsetUp, gridBagConstraints);

        sogGridOffsetDown.setIcon(Resources.getResource(getClass(), "IconDecrement.gif"));
        sogGridOffsetDown.setToolTipText("Move grid line down or left");
        sogGridOffsetDown.setMargin(new java.awt.Insets(0, 0, 0, 0));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 4, 4);
        gridFixed.add(sogGridOffsetDown, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 10);
        individualLayers.add(gridFixed, gridBagConstraints);

        gridArbitrary.setBorder(javax.swing.BorderFactory.createTitledBorder("Arbitrary Grid Control"));
        gridArbitrary.setLayout(new java.awt.GridBagLayout());

        sogDrawGrid.setIcon(Resources.getResource(getClass(), "IconDraw.gif"));
        sogDrawGrid.setToolTipText("Draw grid line");
        sogDrawGrid.setMargin(new java.awt.Insets(0, 0, 0, 0));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 7;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        gridArbitrary.add(sogDrawGrid, gridBagConstraints);

        sogGridIndScroll.setMinimumSize(new java.awt.Dimension(50, 24));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridheight = 6;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 0.5;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 4, 4);
        gridArbitrary.add(sogGridIndScroll, gridBagConstraints);

        sogGridAllUp.setIcon(Resources.getResource(getClass(), "IconIncrement.gif"));
        sogGridAllUp.setToolTipText("Move all grid lines up or right");
        sogGridAllUp.setMargin(new java.awt.Insets(0, 0, 0, 0));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 0, 4);
        gridArbitrary.add(sogGridAllUp, gridBagConstraints);

        sogGridAllDown.setIcon(Resources.getResource(getClass(), "IconDecrement.gif"));
        sogGridAllDown.setToolTipText("Move all grid lines down or left");
        sogGridAllDown.setMargin(new java.awt.Insets(0, 0, 0, 0));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 4, 4);
        gridArbitrary.add(sogGridAllDown, gridBagConstraints);

        sogGridIndUp.setIcon(Resources.getResource(getClass(), "IconIncrement.gif"));
        sogGridIndUp.setToolTipText("Move grid line up or right");
        sogGridIndUp.setMargin(new java.awt.Insets(0, 0, 0, 0));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 0, 4);
        gridArbitrary.add(sogGridIndUp, gridBagConstraints);

        sogGridIndDown.setIcon(Resources.getResource(getClass(), "IconDecrement.gif"));
        sogGridIndDown.setToolTipText("Move grid line down or left");
        sogGridIndDown.setMargin(new java.awt.Insets(0, 0, 0, 0));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 4, 4);
        gridArbitrary.add(sogGridIndDown, gridBagConstraints);

        sogGridIndDelete.setIcon(Resources.getResource(getClass(), "IconDelete.gif"));
        sogGridIndDelete.setToolTipText("Delete grid line");
        sogGridIndDelete.setMargin(new java.awt.Insets(0, 0, 0, 0));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        gridArbitrary.add(sogGridIndDelete, gridBagConstraints);

        sogGridIndNew.setIcon(Resources.getResource(getClass(), "IconNew.gif"));
        sogGridIndNew.setToolTipText("Make new grid line");
        sogGridIndNew.setMargin(new java.awt.Insets(0, 0, 0, 0));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        gridArbitrary.add(sogGridIndNew, gridBagConstraints);

        jLabel3.setText("All:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 4, 0);
        gridArbitrary.add(jLabel3, gridBagConstraints);

        jLabel4.setText("Current:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 4, 0);
        gridArbitrary.add(jLabel4, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 0, 10);
        individualLayers.add(gridArbitrary, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        getContentPane().add(individualLayers, gridBagConstraints);

        entireCell.setBorder(javax.swing.BorderFactory.createTitledBorder("For Entire Cell"));
        entireCell.setLayout(new java.awt.GridBagLayout());

        sogNoSteinerTrees.setText("Do not make Steiner Trees (already done)");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        entireCell.add(sogNoSteinerTrees, gridBagConstraints);

        sogHorVerChoices.add(sogHorOddVerEven);
        sogHorOddVerEven.setText("Horizontal: M1,3,5...   Vertical: M2,4,6...");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 20, 0, 4);
        entireCell.add(sogHorOddVerEven, gridBagConstraints);

        sogHorVerChoices.add(sogVerOddHorEven);
        sogVerOddHorEven.setText("Vertical: M1,3,5...   Horizontal: M2,4,6...");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 20, 0, 4);
        entireCell.add(sogVerOddHorEven, gridBagConstraints);

        sogHorVerChoices.add(sogNoHorVer);
        sogNoHorVer.setText("No Horizontal/Vertical Requirement");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 20, 4, 4);
        entireCell.add(sogNoHorVer, gridBagConstraints);

        jLabel7.setText("Horizontal/Vertical Layer Properties:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 0, 4);
        entireCell.add(jLabel7, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        getContentPane().add(entireCell, gridBagConstraints);

        pack();
    }// </editor-fold>//GEN-END:initComponents

	private void ok(java.awt.event.ActionEvent evt)//GEN-FIRST:event_ok
	{//GEN-HEADEREND:event_ok
		done();
		closeDialog(null);
	}//GEN-LAST:event_ok

	/** Closes the dialog */
	private void closeDialog(java.awt.event.WindowEvent evt)//GEN-FIRST:event_closeDialog
	{
		setVisible(false);
		dispose();

	}//GEN-LAST:event_closeDialog

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel entireCell;
    private javax.swing.JPanel gridArbitrary;
    private javax.swing.JPanel gridControl;
    private javax.swing.JPanel gridFixed;
    private javax.swing.JPanel individualLayers;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JLabel jLabel7;
    private javax.swing.JLabel jLabel8;
    private javax.swing.JButton ok;
    private javax.swing.JScrollPane sogArcList;
    private javax.swing.JCheckBox sogAvoidLayer;
    private javax.swing.JButton sogDrawGrid;
    private javax.swing.JCheckBox sogFavorLayer;
    private javax.swing.JTextField sogFixedOffset;
    private javax.swing.JTextField sogFixedSpacing;
    private javax.swing.JButton sogGridAllDown;
    private javax.swing.JButton sogGridAllUp;
    private javax.swing.JRadioButton sogGridArbitrary;
    private javax.swing.JRadioButton sogGridFixed;
    private javax.swing.JButton sogGridIndDelete;
    private javax.swing.JButton sogGridIndDown;
    private javax.swing.JButton sogGridIndNew;
    private javax.swing.JScrollPane sogGridIndScroll;
    private javax.swing.JButton sogGridIndUp;
    private javax.swing.JLabel sogGridLabel;
    private javax.swing.JRadioButton sogGridNone;
    private javax.swing.JButton sogGridOffsetDown;
    private javax.swing.JButton sogGridOffsetUp;
    private javax.swing.JButton sogGridSpacingDown;
    private javax.swing.JButton sogGridSpacingUp;
    private javax.swing.ButtonGroup sogGridType;
    private javax.swing.JRadioButton sogHorOddVerEven;
    private javax.swing.ButtonGroup sogHorVerChoices;
    private javax.swing.JRadioButton sogNoHorVer;
    private javax.swing.JCheckBox sogNoSteinerTrees;
    private javax.swing.JCheckBox sogShowGrid;
    private javax.swing.JComboBox sogTechList;
    private javax.swing.JRadioButton sogVerOddHorEven;
    // End of variables declaration//GEN-END:variables

}
