/****************************************************************************
**
** Copyright (C) 2015 The Qt Company Ltd.
** Contact: http://www.qt.io/licensing
**
** This file is part of Qt Creator.
**
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company.  For licensing terms and
** conditions see http://www.qt.io/terms-conditions.  For further information
** use the contact form at http://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 or version 3 as published by the Free
** Software Foundation and appearing in the file LICENSE.LGPLv21 and
** LICENSE.LGPLv3 included in the packaging of this file.  Please review the
** following information to ensure the GNU Lesser General Public License
** requirements will be met: https://www.gnu.org/licenses/lgpl.html and
** http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, The Qt Company gives you certain additional
** rights.  These rights are described in The Qt Company LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
****************************************************************************/

#include "windowsupport.h"

#include "actionmanager/actioncontainer.h"
#include "actionmanager/actionmanager.h"
#include "coreconstants.h"
#include "icore.h"

#include <utils/hostosinfo.h>
#include <utils/qtcassert.h>

#include <QAction>
#include <QEvent>
#include <QMenu>
#include <QWidget>

namespace Core {
namespace Internal {


QMenu *WindowList::m_dockMenu = 0;
QList<QWidget *> WindowList::m_windows;
QList<QAction *> WindowList::m_windowActions;
QList<Id> WindowList::m_windowActionIds;

WindowSupport::WindowSupport(QWidget *window, const Context &context)
    : QObject(window),
      m_window(window)
{
    m_window->installEventFilter(this);

    m_contextObject = new IContext(this);
    m_contextObject->setWidget(window);
    m_contextObject->setContext(context);
    ICore::addContextObject(m_contextObject);

    if (UseMacShortcuts) {
        m_minimizeAction = new QAction(this);
        ActionManager::registerAction(m_minimizeAction, Constants::MINIMIZE_WINDOW, context);
        connect(m_minimizeAction, SIGNAL(triggered()), m_window, SLOT(showMinimized()));

        m_zoomAction = new QAction(this);
        ActionManager::registerAction(m_zoomAction, Constants::ZOOM_WINDOW, context);
        connect(m_zoomAction, SIGNAL(triggered()), m_window, SLOT(showMaximized()));

        m_closeAction = new QAction(this);
        ActionManager::registerAction(m_closeAction, Constants::CLOSE_WINDOW, context);
        connect(m_closeAction, SIGNAL(triggered()), m_window, SLOT(close()), Qt::QueuedConnection);
    }

    m_toggleFullScreenAction = new QAction(this);
    updateFullScreenAction();
    ActionManager::registerAction(m_toggleFullScreenAction, Constants::TOGGLE_FULLSCREEN, context);
    connect(m_toggleFullScreenAction, SIGNAL(triggered()), this, SLOT(toggleFullScreen()));

    WindowList::addWindow(window);
}

WindowSupport::~WindowSupport()
{
    if (UseMacShortcuts) {
        ActionManager::unregisterAction(m_minimizeAction, Constants::MINIMIZE_WINDOW);
        ActionManager::unregisterAction(m_zoomAction, Constants::ZOOM_WINDOW);
        ActionManager::unregisterAction(m_closeAction, Constants::CLOSE_WINDOW);
    }
    ActionManager::unregisterAction(m_toggleFullScreenAction, Constants::TOGGLE_FULLSCREEN);
    ICore::removeContextObject(m_contextObject);
    WindowList::removeWindow(m_window);
}

void WindowSupport::setCloseActionEnabled(bool enabled)
{
    if (UseMacShortcuts)
        m_closeAction->setEnabled(enabled);
}

bool WindowSupport::eventFilter(QObject *obj, QEvent *event)
{
    if (obj != m_window)
        return false;
    if (event->type() == QEvent::WindowStateChange) {
        if (Utils::HostOsInfo::isMacHost()) {
            bool minimized = m_window->isMinimized();
            m_minimizeAction->setEnabled(!minimized);
            m_zoomAction->setEnabled(!minimized);
        }
        updateFullScreenAction();
    } else if (event->type() == QEvent::WindowActivate) {
        WindowList::setActiveWindow(m_window);
    }
    return false;
}

void WindowSupport::toggleFullScreen()
{
    if (m_window->isFullScreen()) {
        m_window->setWindowState(m_window->windowState() & ~Qt::WindowFullScreen);
    } else {
        m_window->setWindowState(m_window->windowState() | Qt::WindowFullScreen);
    }
}

void WindowSupport::updateFullScreenAction()
{
    if (m_window->isFullScreen()) {
        if (Utils::HostOsInfo::isMacHost())
            m_toggleFullScreenAction->setText(tr("Exit Full Screen"));
        else
            m_toggleFullScreenAction->setChecked(true);
    } else {
        if (Utils::HostOsInfo::isMacHost())
            m_toggleFullScreenAction->setText(tr("Enter Full Screen"));
        else
            m_toggleFullScreenAction->setChecked(false);
    }
}

void WindowList::addWindow(QWidget *window)
{
#ifdef Q_OS_OSX
    if (!m_dockMenu) {
        m_dockMenu = new QMenu;
        m_dockMenu->setAsDockMenu();
    }
#endif

    m_windows.append(window);
    Id id = Id("QtCreator.Window.").withSuffix(m_windows.size());
    m_windowActionIds.append(id);
    auto action = new QAction(window->windowTitle(), 0);
    m_windowActions.append(action);
    QObject::connect(action, &QAction::triggered, [action]() { WindowList::activateWindow(action); });
    action->setCheckable(true);
    action->setChecked(false);
    Command *cmd = ActionManager::registerAction(action, id,
                                                 Context(Constants::C_GLOBAL));
    cmd->setAttribute(Command::CA_UpdateText);
    ActionManager::actionContainer(Constants::M_WINDOW)->addAction(cmd, Constants::G_WINDOW_LIST);
    QObject::connect(window, &QWidget::windowTitleChanged, [window]() { WindowList::updateTitle(window); });
    if (m_dockMenu)
        m_dockMenu->addAction(action);
    if (window->isActiveWindow())
        setActiveWindow(window);
}

void WindowList::activateWindow(QAction *action)
{
    int index = m_windowActions.indexOf(action);
    QTC_ASSERT(index >= 0, return);
    QTC_ASSERT(index < m_windows.size(), return);
    ICore::raiseWindow(m_windows.at(index));
}

void WindowList::updateTitle(QWidget *window)
{
    int index = m_windows.indexOf(window);
    QTC_ASSERT(index >= 0, return);
    QTC_ASSERT(index < m_windowActions.size(), return);
    QString title = window->windowTitle();
    if (title.endsWith(QStringLiteral("- Qt Creator")))
        title.chop(12);
    m_windowActions.at(index)->setText(title.trimmed());
}

void WindowList::removeWindow(QWidget *window)
{
    // remove window from list,
    // remove last action from menu(s)
    // and update all action titles, starting with the index where the window was
    int index = m_windows.indexOf(window);
    QTC_ASSERT(index >= 0, return);

    ActionManager::unregisterAction(m_windowActions.last(), m_windowActionIds.last());
    delete m_windowActions.takeLast();
    m_windowActionIds.removeLast();

    m_windows.removeOne(window);

    for (int i = index; i < m_windows.size(); ++i)
        updateTitle(m_windows.at(i));
}

void WindowList::setActiveWindow(QWidget *window)
{
    for (int i = 0; i < m_windows.size(); ++i)
        m_windowActions.at(i)->setChecked(m_windows.at(i) == window);
}

} // Internal
} // Core
